# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for the proxy-aware webclient."""

from twisted.application import internet, service
from twisted.internet import ssl
from twisted.web import http, server


# Some settings are not used as described in:
# https://bugzilla.gnome.org/show_bug.cgi?id=648237

TEMPLATE_GSETTINGS_OUTPUT = """\
org.gnome.system.proxy autoconfig-url '{autoconfig_url}'
org.gnome.system.proxy ignore-hosts {ignore_hosts:s}
org.gnome.system.proxy mode '{mode}'
org.gnome.system.proxy.ftp host '{ftp_host}'
org.gnome.system.proxy.ftp port {ftp_port}
org.gnome.system.proxy.http authentication-password '{auth_password}'
org.gnome.system.proxy.http authentication-user '{auth_user}'
org.gnome.system.proxy.http host '{http_host}'
org.gnome.system.proxy.http port {http_port}
org.gnome.system.proxy.http use-authentication {http_use_auth}
org.gnome.system.proxy.https host '{https_host}'
org.gnome.system.proxy.https port {https_port}
org.gnome.system.proxy.socks host '{socks_host}'
org.gnome.system.proxy.socks port {socks_port}
"""

BASE_GSETTINGS_VALUES = {
    "autoconfig_url": "",
    "ignore_hosts": ["localhost", "127.0.0.0/8"],
    "mode": "none",
    "ftp_host": "",
    "ftp_port": 0,
    "auth_password": "",
    "auth_user": "",
    "http_host": "",
    "http_port": 0,
    "http_use_auth": "false",
    "https_host": "",
    "https_port": 0,
    "socks_host": "",
    "socks_port": 0,
}


class SaveHTTPChannel(http.HTTPChannel):
    """A save protocol to be used in tests."""

    protocolInstance = None

    # pylint: disable=C0103
    def connectionMade(self):
        """Keep track of the given protocol."""
        SaveHTTPChannel.protocolInstance = self
        http.HTTPChannel.connectionMade(self)


class SaveSite(server.Site):
    """A site that let us know when it closed."""

    protocol = SaveHTTPChannel

    def __init__(self, *args, **kwargs):
        """Create a new instance."""
        server.Site.__init__(self, *args, **kwargs)
        # we disable the timeout in the tests, we will deal with it manually.
        # pylint: disable=C0103
        self.timeOut = None


class BaseMockWebServer(object):
    """A mock webserver for testing"""

    def __init__(self, ssl_settings=None):
        """Start up this instance."""
        self.root = self.get_root_resource()
        self.site = SaveSite(self.root)
        application = service.Application('web')
        self.service_collection = service.IServiceCollection(application)
        #pylint: disable=E1101
        ssl_context = None
        if (ssl_settings is not None
                and 'key' in ssl_settings
                and 'cert' in ssl_settings):
            ssl_context = ssl.DefaultOpenSSLContextFactory(ssl_settings['key'],
                                                         ssl_settings['cert'])
            self.ssl_server = internet.SSLServer(0, self.site, ssl_context)
        else:
            self.ssl_server = None
        self.server = internet.TCPServer(0, self.site)
        self.server.setServiceParent(self.service_collection)
        if self.ssl_server:
            self.ssl_server.setServiceParent(self.service_collection)
        self.service_collection.startService()

    def get_root_resource(self):
        """Get the root resource with all the children."""
        raise NotImplementedError

    def get_iri(self):
        """Build the iri for this mock server."""
        url = u"http://127.0.0.1:%d/"
        return url % self.get_port()

    def get_ssl_iri(self):
        """Build the ssl iri for this mock server."""
        if self.ssl_server:
            url = u"https://127.0.0.1:%d/"
            return url % self.get_ssl_port()

    def get_port(self):
        """Return the port where we are listening."""
        # pylint: disable=W0212
        return self.server._port.getHost().port
        # pylint: enable=W0212

    def get_ssl_port(self):
        """Return the ssl port where we are listening."""
        # pylint: disable=W0212
        if self.ssl_server:
            return self.ssl_server._port.getHost().port
        # pylint: enable=W0212

    def stop(self):
        """Shut it down."""
        #pylint: disable=E1101
        if self.site.protocol.protocolInstance:
            self.site.protocol.protocolInstance.timeoutConnection()
        return self.service_collection.stopService()
