# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Confirm some actions."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'AreYouSure',
    'CleanupProblem',
    ]


from computerjanitorapp import setup_gettext
from gi.repository import Gtk
from operator import mod as interpolate


_ = setup_gettext()
NL = '\n'


class DialogBase:
    """Base class for dialogs."""

    def __init__(self, ui):
        self._ui = ui


class AreYouSure(DialogBase):
    """Confirmation of destructive actions."""

    def verify(self):
        """Confirm package removal."""
        # Start by getting all the active, non-ignored cruft.  These are the
        # candidates for removal.
        cleanable_cruft = self._ui.get_cleanable_cruft()
        package_cruft_count = sum(1 for cruft_name, is_pkg in cleanable_cruft
                                  if is_pkg)
        other_cruft_count = len(cleanable_cruft) - package_cruft_count
        messages = []
        ok_button = None
        # XXX vv barry 2010-08-23: LP: #622720
        if package_cruft_count > 0:
            messages = [
                _('<b>Software packages to remove: %(packages)s</b>.'),
                _('\nRemoving packages that are still in use can '
                  'cause errors.'),
                ]
            ok_button = _('Remove packages')
        if other_cruft_count > 0:
            messages.insert(1, _('Non-package items to remove: %(others)s.'))
            ok_button = _('Clean up')
        message = interpolate(NL.join(messages),
                              dict(packages=package_cruft_count,
                                   others=other_cruft_count))
        # XXX ^^ barry 2010-08-23: LP: #622720
        if ok_button is None:
            # The user de-selected all cruft from the ui, so there's actually
            # nothing to clean up.
            dialog = Gtk.MessageDialog(
                parent=self._ui.widgets['window'],
                type=Gtk.MessageType.WARNING,
                buttons=Gtk.ButtonsType.NONE,
                message_format=_('There is nothing to clean up'),
                )
            dialog.set_title(_('Clean up'))
            dialog.format_secondary_markup(message)
            dialog.add_button(_('Ok'), Gtk.ResponseType.YES)
            dialog.show_all()
            dialog.run()
            dialog.hide()
            return False
        # It would be nice if we could produce better messages than these,
        # but that would require a richer interface to the dbus service,
        # and probably to the cruft plugin architecture underneath that.
        message = _('Are you sure you want to clean your system?')
        dialog = Gtk.MessageDialog(
            parent=self._ui.widgets['window'],
            type=Gtk.MessageType.WARNING,
            buttons=Gtk.ButtonsType.NONE,
            message_format=message)
        dialog.set_title(_('Clean up'))
        dialog.format_secondary_markup(message)
        dialog.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CLOSE)
        dialog.add_button(ok_button, Gtk.ResponseType.YES)
        # Show the dialog and get the user's response.
        dialog.show_all()
        response = dialog.run()
        dialog.hide()
        return response == Gtk.ResponseType.YES


class CleanupProblem(DialogBase):
    """Some problem occurred during cleanup.

    This is almost definitely caused by another package manager running at the
    same time.
    """

    def run(self):
        """Show the dialog and wait for confirmation."""
        message = _('System clean up could not complete. '
                    'Be sure no other package manager such as Synaptic or '
                    'Update Manager is running.')
        dialog = Gtk.MessageDialog(
            parent=self._ui.widgets['window'],
            type=Gtk.MessageType.ERROR,
            buttons=Gtk.ButtonsType.OK,
            message_format=message)
        dialog.set_title(_('System clean up failure'))
        dialog.show_all()
        dialog.run()
        dialog.hide()
