# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Authentication for Computer Janitor backend services."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'Authenticator',
    ]


import dbus


PK_AUTHORITY_BUS_NAME = 'org.freedesktop.PolicyKit1'
PK_AUTHORITY_OBJECT_PATH = '/org/freedesktop/PolicyKit1/Authority'
PK_AUTHORITY_INTERFACE = 'org.freedesktop.PolicyKit1.Authority'
# From the PolicyKit API.
# http://hal.freedesktop.org/docs/polkit/
# eggdbus-interface-org.freedesktop.PolicyKit1.Authority.html
AllowUserInteraction = 0x00000001


class Authenticator:
    """PolicyKit authenticator."""

    def authenticate(self, sender, connection, privilege):
        """Authenticate with PolicyKit.

        :param sender: The initiator of the action.
        :param connection: The dbus connection that initiated the action.
        :param privilege: The privilege being requested.
        :return: Whether the subject is authorized or not.
        :rtype: bool
        """
        policykit = self._get_policykit_proxy()
        sender_pid = self._get_sender_pid(connection, sender)
        # This is the CheckAuthorization() 'subject' structure.
        subject = (
            'unix-process', {
                'pid': sender_pid,
                'start-time': dbus.UInt64(0),
                })
        # No details or cancellation_id needed.
        details = {'': ''}
        cancellation_id = ''
        flags = AllowUserInteraction
        # CheckAuthorization returns an AuthorizationResult structure, modeled
        # as a 3-tuple.  The only thing we care about though is the boolean
        # describing whether we got authorized or not.
        is_authorized, is_challenge, details = policykit.CheckAuthorization(
            subject, privilege, details, flags, cancellation_id)
        return is_authorized

    def _get_policykit_proxy(self):
        """Contact the system bus to get a PolicyKit proxy."""
        system_bus = dbus.SystemBus()
        pk_proxy = system_bus.get_object(
            PK_AUTHORITY_BUS_NAME, PK_AUTHORITY_OBJECT_PATH)
        return dbus.Interface(pk_proxy, PK_AUTHORITY_INTERFACE)

    def _get_sender_pid(self, connection, sender):
        """Contact the system bus to get the sender connection PID."""
        # Since we're going to authorize a Unix process, we need to get the
        # sender's process id.  This is available on the dbus.  The
        # CheckAuthorization() method also requires us to have a start-time,
        # but it's not clear what the semantics are for that, so we'll just
        # put a zero there.
        db_proxy = connection.get_object(
            dbus.BUS_DAEMON_NAME, dbus.BUS_DAEMON_PATH, introspect=False)
        info = dbus.Interface(db_proxy, dbus.BUS_DAEMON_IFACE)
        return info.GetConnectionUnixProcessID(sender)
