/*
Copyright (c) 2009-2010 Sony Pictures Imageworks Inc., et al.
All Rights Reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:
* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.
* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.
* Neither the name of Sony Pictures Imageworks nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/** Lexical scanner for OpenShadingLanguage 'object' files
 **/


/************************************************************
 * Definitions section
 ************************************************************/


/* Option 'noyywrap' indicates that when EOF is hit, yyin does not 
 * automatically reset to another file.
 */
%option noyywrap

 /* Option 'prefix' creates a C++ lexer with the given prefix, so that
  * we can link with other flex-generated lexers in the same application
  * without name conflicts.
  */
%option prefix="oso"


 /* Define regular expression macros 
  ************************************************/

 /* white space, not counting newline */
WHITE           [ \t\v\f\r]+
 /* alpha character */
ALPHA           [A-Za-z]
 /* numerals */
DIGIT           [0-9]
 /* Integer literal */
INTEGER         [-+]?{DIGIT}+
 /* floating point literal (E, FLT1, FLT2, FLT3 are just helpers) */
E               [eE][-+]?{DIGIT}+
FLT1            [-+]?{DIGIT}+\.{DIGIT}*{E}?
FLT2            [-+]?{DIGIT}*\.{DIGIT}+{E}?
FLT3            [-+]?{DIGIT}+{E}
FLT             {FLT1}|{FLT2}|{FLT3}
 /* string literal */
STR     \"(\\.|[^\\"])*\"
 /* Identifier: alphanumeric, may contain digits after the first character.
  * Also '$' and '.' are allowed!
  */
IDENT           ({ALPHA}|[_$])({ALPHA}|{DIGIT}|[_$\.])*
 /* C preprocessor (cpp) directives */
COMMENT         \#[^\n]*\n
 /* Hints */
HINTPATTERN     \%{IDENT}(\{([^\}\\"]|{STR})*\})?


 /* Note for lex newbies: the following '%{ .. %}' section contains literal
  * C code that will be inserted at the top of code that flex generates.
  */
%{
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <vector>
#include <string>

#include <OpenImageIO/ustring.h>
#include <OpenImageIO/strutil.h>

#include "osoreader.h"
using namespace OSL;
using namespace OSL::pvt;

#include "osogram.hpp"   /* Generated by bison/yacc */

#define yylval osolval

#ifdef _WIN32
#define YY_NO_UNISTD_H
#endif

%}

/* Declare modes */
%s DECLARATION


%%

 /************************************************
  * Lexical matching rules
  ************************************************/

 /* Comments */
{COMMENT}               {  /* skip it */ }

 /* keywords */
<DECLARATION>"closure"	{  return (yylval.i=CLOSURE); }
<DECLARATION>"color"	{  return (yylval.i=COLORTYPE); }
<DECLARATION>"float"	{  return (yylval.i=FLOATTYPE); }
<DECLARATION>"int"      {  return (yylval.i=INTTYPE); }
<DECLARATION>"matrix"	{  return (yylval.i=MATRIXTYPE); }
<DECLARATION>"normal"	{  return (yylval.i=NORMALTYPE); }
<DECLARATION>"point"	{  return (yylval.i=POINTTYPE); }
<DECLARATION>"string"	{  return (yylval.i=STRINGTYPE); }
<DECLARATION>"struct"	{  return (yylval.i=STRUCT); }
<DECLARATION>"vector"	{  return (yylval.i=VECTORTYPE); }

^local                  {
                           BEGIN (DECLARATION);
                           yylval.i = SymTypeLocal;
                           return SYMTYPE;
                        }

^temp                   {
                           BEGIN (DECLARATION);
                           yylval.i = SymTypeTemp;
                           return SYMTYPE;
                        }

^global                 {
                           BEGIN (DECLARATION);
                           yylval.i = SymTypeGlobal;
                           return SYMTYPE;
                        }

^param                  {
                           BEGIN (DECLARATION);
                           yylval.i = SymTypeParam;
                           return SYMTYPE;
                        }

^oparam                 {
                            BEGIN (DECLARATION);
                            yylval.i = SymTypeOutputParam;
                            return SYMTYPE;
                        }

^const                  {
                            BEGIN (DECLARATION);
                            yylval.i = SymTypeConst;
                            return SYMTYPE;
                        }

^code                   {
                            BEGIN (INITIAL);
                            return yylval.i = CODE;
                        }

 /* Identifiers */
{IDENT}	                {
                            yylval.s = ustring(YYText()).c_str();
                            // std::cerr << "lex ident '" << yylval.s << "'\n";
                            return IDENTIFIER;
                        }

 /* Literal values */
{INTEGER}               {
                            yylval.i = atoi (YYText());
                            // std::cerr << "lex int " << yylval.i << "\n";
                            return INT_LITERAL;
                        }

{FLT}                   {
                            yylval.f = atof (YYText());
                            // std::cerr << "lex float " << yylval.f << "\n";
                            return FLOAT_LITERAL;
                        }

{STR}                   {
                            // grab the material between the quotes
                            std::string escaped (YYText(), 1, strlen(YYText())-2);
                            ustring s (OIIO::Strutil::unescape_chars(escaped));
                            yylval.s = s.c_str();
                            // std::cerr << "lex string '" << yylval.s << "'\n";
                            return STRING_LITERAL;
                        }

{HINTPATTERN}           {
                            ustring s (YYText());
                            yylval.s = s.c_str();
                            return HINT;
                        }

 /* Ignore whitespace */
{WHITE} 		{  }

 /* Ignore a linefeed that ends with a trailing blackslash */
"\\\n"		        {  }

 /* End of line */
[\n]			{
                            OSOReader::osoreader->incr_lineno ();
                            return ENDOFLINE;
                        }

 /* catch-all rule for any other single characters */
.			{  return (yylval.i = *YYText()); }

%%


