#********************************************************************** 
# 
#   Danny.OOo.WindowLib.py 
# 
#   A module to easily work with OpenOffice.org. 
# 
#********************************************************************** 
#   Copyright (c) 2003-2004 Danny Brewer 
#   d29583@groovegarden.com 
# 
#   This library is free software; you can redistribute it and/or 
#   modify it under the terms of the GNU Lesser General Public 
#   License as published by the Free Software Foundation; either 
#   version 2.1 of the License, or (at your option) any later version. 
# 
#   This library is distributed in the hope that it will be useful, 
#   but WITHOUT ANY WARRANTY; without even the implied warranty of 
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
#   Lesser General Public License for more details. 
# 
#   You should have received a copy of the GNU Lesser General Public 
#   License along with this library; if not, write to the Free Software 
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
# 
#   See:  http://www.gnu.org/licenses/lgpl.html 
# 
#********************************************************************** 
#   If you make changes, please append to the change log below. 
# 
#   Change Log 
#   Danny Brewer         Revised 2004-06-05-02 
# 
#********************************************************************** 



# OOo's libraries 
import uno 
try:
  import unohelper 
except:
  pass
# Danny's libraries 
from doooLib import createUnoService, createUnoStruct 
from doooLib import makeRectangle, StarDesktop 
# 
from doooListenersProcAdapters import * 


#---------- 
#   com.sun.star.awt.WindowAttribute 
# 
# specifies that the window is initially visible. 
com_sun_star_awt_WindowAttribute_SHOW        = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.SHOW" ) 
# specifies that the window fills the complete desktop area. 
com_sun_star_awt_WindowAttribute_FULLSIZE    = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.FULLSIZE" ) 
com_sun_star_awt_WindowAttribute_OPTIMUMSIZE = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.OPTIMUMSIZE" ) 
com_sun_star_awt_WindowAttribute_MINSIZE     = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.MINSIZE" ) 
# specifies that the window has visible borders. 
com_sun_star_awt_WindowAttribute_BORDER      = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.BORDER" ) 
# specifies that the size of the window can be changed by the user. 
com_sun_star_awt_WindowAttribute_SIZEABLE    = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.SIZEABLE" ) 
# specifies that the window can be moved by the user. 
com_sun_star_awt_WindowAttribute_MOVEABLE    = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.MOVEABLE" ) 
# specifies that the window can be closed by the user. 
com_sun_star_awt_WindowAttribute_CLOSEABLE   = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.CLOSEABLE" ) 
#[ DEPRECATED ] specifies that the window should support the XSystemDependentWindowPeer interface. 
com_sun_star_awt_WindowAttribute_SYSTEMDEPENDENT = uno.getConstantByName( "com.sun.star.awt.WindowAttribute.SYSTEMDEPENDENT" ) 


# Default window attributes. 
# The DBWindow class (and subclasses) use this as the default window attributes 
#  when you don't specify any window attributes on the constructor. 
gnDefaultWindowAttributes = com_sun_star_awt_WindowAttribute_SHOW + com_sun_star_awt_WindowAttribute_BORDER + com_sun_star_awt_WindowAttribute_MOVEABLE + com_sun_star_awt_WindowAttribute_CLOSEABLE 
        # we could have addedd... 
        #com_sun_star_awt_WindowAttribute_SIZEABLE 



# The global Awt Toolkit. 
# This is initialized the first time it is needed. 
#goAwtToolkit = createUnoService( "com.sun.star.awt.Toolkit" ) 
goAwtToolkit = None 
# 
def getAwtToolkit(): 
    global goAwtToolkit 
    if goAwtToolkit == None: 
        goAwtToolkit = createUnoService( "com.sun.star.awt.Toolkit" ) 
    return goAwtToolkit 



# This class is a sugar coated abstraction for OOo's Awt Window. 
# This can be used in two different ways... 
# 1. by subclassing it (elegant) 
# 2. without subclassing it (less elegant) 
class DBWindow: 
    """Class to build a modeless window from the com.sun.star.awt.* services. 
    This doesn't do anything you couldn't already do using OOo's UNO API, 
     this just makes it much easier. 
    This class IS NOT the actual OOo window, but you can get the OOo window or frame by calling getWindow() or getFrame() on this class. 
    You can change the window size, position, title, etc. 
    You can add controls, and listeners for those controls to the window. 
    This class can be used by subclassing it, or without subclassing it. 
    """ 
    def __init__( self, cTitle=None, tBoundsRect=None, nWindowAttributes=None ): 
        oAwtToolkit = getAwtToolkit() 
        oWindowDesc = self.createMyWindowDescriptor( tBoundsRect, nWindowAttributes ) 
        self.oWindow = oAwtToolkit.createWindow( oWindowDesc ) 
        
        # At this point, if you stop the program, you will have a 
        #  new OOo window on the screen, but you cannot do anything 
        #  with it.  You cannot even close it! 
        # In fact, you have to kill the OOo process, as even OOo 
        #  cannot close this window since it has no Frame and 
        #  therefore is not integrated into the desktop environment. 
        # So we need to create a Frame for our window.... 

        # Create a new frame. 
        self.oFrame = createUnoService( "com.sun.star.frame.Frame" ) 
        # Initialize this frame with our new window. 
        self.oFrame.initialize( self.oWindow ) 
        # Tell the frame that its parent is the Desktop. 
        self.oFrame.setCreator( StarDesktop ) 

        if cTitle != None: 
            self.setWindowTitle( cTitle ) 

        # A collection of controls on this window.  (NOT control models, but controls.) 
        self.controls = {} 

    def createMyWindowDescriptor( self, tBoundsRect=None, nWindowAttributes=None ): 
        """This returns the com.sun.star.awt.WindowDescriptor that will be used to create this window. 
        Your subclass can override this method to provide a modified window descriptor.""" 
        if tBoundsRect == None: 
            tBoundsRect = makeRectangle( 100, 200, 300, 400 ) 
        if nWindowAttributes == None: 
            nWindowAttributes = gnDefaultWindowAttributes 
        oAwtToolkit = getAwtToolkit() 
        
        oWindowDesc = createUnoStruct( "com.sun.star.awt.WindowDescriptor" ) 
        
        # specifies a top level window on the desktop. It is also a container. 
        # LliureX Platinum
        #oWindowDesc.Type = uno.getConstantByName( "com.sun.star.awt.WindowClass.TOP" ) 
        
        # specifies the name of the component service. 
        # A zero length name means that the vcl creates a blank top, 
        #  a container, or a simple window. 
        oWindowDesc.WindowServiceName = "" 
        
        # specifies the parent of the component. 
        # If Parent == 0 && ParentIndex == -1 , then the window is on the desktop. 
        oWindowDesc.Parent = oAwtToolkit.getDesktopWindow() 
        # specifies the index of the parent window, if available. 
        # If Parent == 0 and this struct is a member of an array, 
        #  then this is the offset from the beginning of the array to the parent. 
        #  A value of -1 means desktop. 
        oWindowDesc.ParentIndex = -1 
        
        # specifies the position and size of the window. 
        # This member is ignored if the window attribute has 
        #  WindowAttribute::FULLSIZE . 
        oWindowDesc.Bounds = tBoundsRect 
        
        # specifies the window attributes. 
        oWindowDesc.WindowAttributes = nWindowAttributes 

        return oWindowDesc 

    #-------------------------------------------------- 
    #   The following two methods 
    #   getWindow() and getFrame() 
    #   return the two objects that this class manages. 
    #-------------------------------------------------- 

    def getWindow( self ): 
        """Return the OOo Awt Window that this object represents. 
        The window object returned has the following interfaces... 
            com.sun.star.awt.XTopWindow 
            com.sun.star.awt.XWindow 
            com.sun.star.awt.XWindowPeer 
            com.sun.star.awt.XVclContainer      [deprecated] 
            com.sun.star.awt.XVclContainerPeer  [deprecated] 
            com.sun.star.awt.XVclWindowPeer     [deprecated] 
            com.sun.star.awt.XLayoutConstraints 
            com.sun.star.awt.XView 
            com.sun.star.awt.XDevice 
            com.sun.star.lang.XEventListener 
            com.sun.star.lang.XComponent 
            com.sun.star.lang.XTypeProvider 
            com.sun.star.accessibility.XAccessible 
        Properties... 
            <object>    MenuBar 
            <array>     Windows 
            <array>     Group 
            <object>    PosSize 
            Boolean     Visible 
            Boolean     Enable 
            <object>    Toolkit 
            <object>    Pointer 
            Long        Background 
            Boolean     DesignMode 
            Long        Foreground 
            <object>    ControlFont 
            <object>    MinimumSize 
            <object>    PreferredSize 
            <object>    AccessibleContext 
            <object>    Graphics 
            <object>    Size 
            <object>    Info 
            <array>     FontDescriptors 
        """ 
        return self.oWindow 
    
    def getFrame( self ): 
        """Return the Frame for the OOo Awt Window that this object represents.""" 
        return self.oFrame 

    #-------------------------------------------------- 
    #   The following methods represent part of the sugar coating 
    #    that this class provides to manage the window. 
    #-------------------------------------------------- 

    def windowClose( self ): 
        """Close the window frame.""" 
        self.getFrame().close( True ) 
        self.oFrame = None 
        self.oWindow = None 

    def windowToFront( self ): 
        """places this window at the top of the stacking order and shows it in front of any other windows.""" 
        # By looking at the following line of code, 
        #  and by looking at the list of interfaces from the comment in the getWindow() method, 
        #  you should be able to figure out how to do lots of other things to this window. 
        self.getWindow().toFront() 

    def windowToBack( self ): 
        """ places this window at the bottom of the stacking order and makes the corresponding adjustment to other visible windows.""" 
        self.getWindow().toBack() 

    def setWindowTitle( self, cTitle ): 
        """Set the title of the window that this object represents.""" 
        # By looking at the following line of code, 
        #  and by looking at the com.sun.star.frame.Frame service, 
        #  you should be able to figure out how to do lots of other things to this window. 
        self.getFrame().Title = cTitle 

    def getWindowTitle( self ): 
        """Get the title of the window that this object represents""" 
        return self.getFrame().Title 

    def setWindowBackground( self, nColor ): 
        """Set the background color of the window that this object represents.""" 
        self.getWindow().setBackground( nColor ) 

    def getWindowGraphics( self ): 
        """Returns a com.sun.star.awt.XGraphics for the window that this object represents. 
        You can then call methods on the XGraphics to draw things into this window.""" 
        return self.getWindow().getGraphics() 

    def setWindowPosSize( self, nX, nY, nWidth, nHeight ): 
        """Set the position and size of the window.""" 
        self.getWindow().setPosSize( nX, nY, nWidth, nHeight, 
                                     uno.getConstantByName( "com.sun.star.awt.PosSize.POSSIZE" ) ) 
    def setWindowPosition( self, nX, nY ): 
        """Set the position of the window.""" 
        self.getWindow().setPosSize( nX, nY, 0, 0, 
                                     uno.getConstantByName( "com.sun.star.awt.PosSize.POS" ) ) 
    def setWindowSize( self, nWidth, nHeight ): 
        """Set the position and size of the window.""" 
        self.getWindow().setPosSize( 0, 0, nWidth, nHeight, 
                                     uno.getConstantByName( "com.sun.star.awt.PosSize.SIZE" ) ) 
    def getWindowPosSize( self ): 
        """Returns a com.sun.star.awt.Rectangle that contains the position and size of this window.""" 
        return self.getWindow().getPosSize() 

    def setWindowVisible( self, bVisible ): 
        """Make this window (in)visible.""" 
        self.getWindow().setVisible( bVisible ) 
    def setWindowEnable( self, bEnable ): 
        """Make this window (dis)enabled.""" 
        self.getWindow().setEnable( bEnable ) 

    def setWindowFocus( self ): 
        """Set the focus to this window.""" 
        self.getWindow().setFocus() 


    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlButton 
    #-------------------------------------------------- 

    def addButton( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        cLabel=None, 
                        actionListenerProc=None ): 
        """Add a Button control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlButtonModel" ) 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if cLabel != None: 
            oControlModel.Label = cLabel 

        if actionListenerProc != None: 
            self.addActionListenerProc( cCtrlName, actionListenerProc ) 

    def setButtonLabel( self, cCtrlName, cLabel ): 
        """Set the label of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setLabel( cLabel ) 

    
    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlCheckBox 
    #-------------------------------------------------- 

    def addCheckBox( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        cLabel=None, 
                        itemListenerProc=None ): 
        """Add a CheckBox control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlCheckBoxModel" ) 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if cLabel != None: 
            oControlModel.Label = cLabel 

        if itemListenerProc != None: 
            self.addItemListenerProc( cCtrlName, itemListenerProc ) 

    def setCheckBoxLabel( self, cCtrlName, cLabel ): 
        """Set the label of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setLabel( cLabel ) 

    def getCheckBoxState( self, cCtrlName ): 
        """Get the state of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getState(); 

    def setCheckBoxState( self, cCtrlName, nState ): 
        """Set the state of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setState( nState ) 

    def enableCheckBoxTriState( self, cCtrlName, bTriStateEnable ): 
        """Enable or disable the tri state mode of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.enableTriState( bTriStateEnable ) 

    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlOptionButton
    #-------------------------------------------------- 

    def addOptionButton( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        cLabel=None, 
                        itemListenerProc=None ): 
        """Add a OptionButton control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlRadioButtonModel" ) 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if cLabel != None: 
            oControlModel.Label = cLabel 

        if itemListenerProc != None: 
            self.addItemListenerProc( cCtrlName, itemListenerProc ) 

    def setOptionButtonLabel( self, cCtrlName, cLabel ): 
        """Set the label of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setLabel( cLabel ) 

    def getOptionButtonState( self, cCtrlName ): 
        """Get the state of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getState(); 

    def setOptionButtonState( self, cCtrlName, nState ): 
        """Set the state of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setState( nState ) 


    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlComboBox 
    #-------------------------------------------------- 

    def addComboBox( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        bDropdown=True, 
                        itemListenerProc=None, 
                        actionListenerProc=None ): 
        """Add a ComboBox control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlComboBoxModel" ) 
        oControlModel.Dropdown = bDropdown 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if itemListenerProc != None: 
            self.addItemListenerProc( cCtrlName, itemListenerProc ) 
        if actionListenerProc != None: 
            self.addActionListenerProc( cCtrlName, actionListenerProc ) 

    def addComboBoxItem( self, cCtrlName, cItemText, nPosition=-1 ): 
        """Add an item to the ComboBox at specified position.""" 
        oControl = self.getControl( cCtrlName ) 
        if nPosition == -1: nPosition = self.getComboBoxItemCount( cCtrlName ) 
        oControl.addItem( cItemText, nPosition ) 

    def addComboBoxItems( self, cCtrlName, tcItemTexts, nPosition=0 ): 
        """Add a tupple of items to the ComboBox at specified position.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.addItems( tcItemTexts, nPosition ) 

    def removeComboBoxItems( self, cCtrlName, nPosition, nCount=1 ): 
        """Remove items from a ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.removeItems( nPosition, nCount ) 

    def getComboBoxItemCount( self, cCtrlName ): 
        """Get the number of items in a ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getItemCount() 

    def getComboBoxItem( self, cCtrlName, nPosition ): 
        """Return the item at specified position within the ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getItem( nPosition ) 

    def getComboBoxItems( self, cCtrlName ): 
        """Return a tupple of all items in the ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getItems() 

    def getComboBoxDropDownLineCount( self, cCtrlName ): 
        """Returns the number of visible lines in the drop down mode.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getDropDownLineCount() 

    def setComboBoxDropDownLineCount( self, cCtrlName, nNumLines ): 
        """Sets the number of visible lines in the drop down mode.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setDropDownLineCount( nNumLines ) 

    def getComboBoxText( self, cCtrlName ): 
        """Get the text of the ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getText(); 

    def setComboBoxText( self, cCtrlName, cText ): 
        """Set the text of the ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setText( cText ) 

    def insertComboBoxText( self, cCtrlName, cText, nMinSelection, nMaxSelection ): 
        """Insert text at specified position in the ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        tSelection = Selection() 
        tSelection.Min = nMinSelection 
        tSelection.Max = nMaxSelection 
        oControl.insertText( tSelection, cText ) 

    def getComboBoxSelectedText( self, cCtrlName ): 
        """Get the selected text of the ComboBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelectedText(); 

    def getComboBoxSelection( self, cCtrlName ): 
        """Get the selection of the ComboBox. 
        This is a com.sun.star.awt.Selection struct with members Min and Max.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelection() 

    def isComboBoxEditable( self, cCtrlName ): 
        """Indicate whether the text in the ComboBox is editable by the user.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.isEditable() 

    def setComboBoxEditable( self, cCtrlName, bEditable ): 
        """Set whether the text in the ComboBox is editable by the user.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setEditable( bEditable ) 

    def setComboBoxMaxTextLen( self, cCtrlName, nMaxLen ): 
        """Set the maximum text length the ComboBox can have.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setMaxTextLen( nMaxLen ) 

    def getComboBoxMaxTextLen( self, cCtrlName ): 
        """Get the maximum text length the ComboBox can have.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.getMaxTextLen() 

    
    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlEdit 
    #-------------------------------------------------- 

    def addEdit( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        cText=None, 
                        textListenerProc=None,keyListenerProc=None): 
        """Add a Edit control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlEditModel" ) 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if cText != None: 
            self.setEditText( cCtrlName, cText ) 
        if textListenerProc != None: 
            self.addTextListenerProc( cCtrlName, textListenerProc )
        if keyListenerProc != None: 
            self.addKeyListenerProc( cCtrlName, keyListenerProc )

    def getEditText( self, cCtrlName ): 
        """Get the text of the edit box.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getText(); 

    def setEditText( self, cCtrlName, cText ): 
        """Set the text of the edit box.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setText( cText ) 

    def insertEditText( self, cCtrlName, cText, nMinSelection, nMaxSelection ): 
        """Insert text at specified position in the edit box.""" 
        oControl = self.getControl( cCtrlName ) 
        tSelection = Selection() 
        tSelection.Min = nMinSelection 
        tSelection.Max = nMaxSelection 
        oControl.insertText( tSelection, cText ) 

    def getEditSelectedText( self, cCtrlName ): 
        """Get the selected text of the edit box.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelectedText(); 

    def getEditSelection( self, cCtrlName ): 
        """Get the selection of the edit box. 
        This is a com.sun.star.awt.Selection struct with members Min and Max.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelection() 

    def isEditEditable( self, cCtrlName ): 
        """Indicate whether the text in the edit box is editable by the user.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.isEditable() 

    def setEditEditable( self, cCtrlName, bEditable ): 
        """Set whether the text in the edit box is editable by the user.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setEditable( bEditable ) 

    def setEditMaxTextLen( self, cCtrlName, nMaxLen ): 
        """Set the maximum text length the edit box can have.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setMaxTextLen( nMaxLen ) 

    def getEditMaxTextLen( self, cCtrlName ): 
        """Get the maximum text length the edit box can have.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.getMaxTextLen() 

        
    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlFixedText 
    #-------------------------------------------------- 

    def addFixedText( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        cLabel=None ): 
        """Add a FixedText control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlFixedTextModel" ) 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if cLabel != None: 
            oControlModel.Label = cLabel 

    def getFixedTextText( self, cCtrlName ): 
        """Get the text of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getText(); 

    def setFixedTextText( self, cCtrlName, cText ): 
        """Set the text of the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setText( cText ) 

    def getFixedTextAlignment( self, cCtrlName ): 
        """Get the alignment of the control.  0:left, 1:center, 2:right""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getAlignment(); 

    def setFixedTextAlignment( self, cCtrlName, nAlignment ): 
        """Set the alignment of the control.  0:left, 1:center, 2:right""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setAlignment( nAlignment ) 

        
    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlImageControl 
    #-------------------------------------------------- 

    def addImageControl( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                         nBorder=1 ): 
        """Add an ImageControl control to the window. 
        nBorder = 0:none, 1:3D border, 2:simple border""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlImageControlModel" ) 
        oControlModel.Border = nBorder 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

    def getImageControlImageURL( self, cCtrlName ): 
        """Get the ImageURL of the control.""" 
        self.getControlModelProperty( cCtrlName, "ImageURL" ) 

    def setImageControlImageURL( self, cCtrlName, cImageURL ): 
        """Get the ImageURL of the control.""" 
        self.setControlModelProperty( cCtrlName, "ImageURL", cImageURL ) 

    def setImageControlBackgroundColor(self,cCtrlName,cImageBackColor):
        self.setControlModelProperty(cCtrlName,"BackgroundColor",cImageBackColor)

    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlListBox 
    #-------------------------------------------------- 

    def addListBox( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        bDropdown=False, 
                        itemListenerProc=None, 
                        actionListenerProc=None ): 
        """Add a ComboBox control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlListBoxModel" ) 
        oControlModel.Dropdown = bDropdown 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if itemListenerProc != None: 
            self.addItemListenerProc( cCtrlName, itemListenerProc ) 
        if actionListenerProc != None: 
            self.addActionListenerProc( cCtrlName, actionListenerProc ) 

    def addListBoxItem( self, cCtrlName, cItemText, nPosition=-1 ): 
        """Add an item to the ListBox at specified position.""" 
        oControl = self.getControl( cCtrlName ) 
        if nPosition == -1: nPosition = self.getListBoxItemCount( cCtrlName ) 
        oControl.addItem( cItemText, nPosition ) 

    def addListBoxItems( self, cCtrlName, tcItemTexts, nPosition=0 ): 
        """Add a tupple of items to the ListBox at specified position.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.addItems( tcItemTexts, nPosition ) 

    def removeListBoxItems( self, cCtrlName, nPosition, nCount=1 ): 
        """Remove items from a ListBox.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.removeItems( nPosition, nCount ) 

    def getListBoxItemCount( self, cCtrlName ): 
        """Get the number of items in a ListBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getItemCount() 

    def getListBoxItem( self, cCtrlName, nPosition ): 
        """Return the item at specified position within the ListBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getItem( nPosition ) 

    def getListBoxItems( self, cCtrlName ): 
        """Return a tupple of all items in the ListBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getItems() 

    def getListBoxSelectedItemPos( self, cCtrlName ): 
        """Returns the position of the currently selected item.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelectedItemPos() 

    def getListBoxSelectedItemsPos( self, cCtrlName ): 
        """Returns the positions of all currently selected items.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelectedItemsPos() 

    def getListBoxSelectedItem( self, cCtrlName ): 
        """Returns the currently selected item.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelectedItem() 

    def getListBoxSelectedItems( self, cCtrlName ): 
        """Returns all currently selected items.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getSelectedItems() 

    def selectListBoxItemPos( self, cCtrlName, nItemPos, bSelect=True ): 
        """Select/Deselect the item at the specified position.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.selectItemPos( nItemPos, bSelect ) 

    def selectListBoxItemsPos( self, cCtrlName, tnItemPos, bSelect=True ): 
        """Selects/Deselects the items at the specified positions.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.selectItemsPos( tnItemPos, bSelect ) 

    def selectListBoxItem( self, cCtrlName, cItemText, bSelect=True ): 
        """Selects/Deselects the ispecified item.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.selectItem( cItemText, bSelect ) 

    def isListBoxMultipleMode( self, cCtrlName ): 
        """Selects/Deselects the ispecified item.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.isMutipleMode() 

    def setListBoxMultipleMode( self, cCtrlName, bMultipleMode ): 
        """Selects/Deselects the ispecified item.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.setMutipleMode( bMultipleMode ) 

    def getListBoxDropDownLineCount( self, cCtrlName ): 
        """Returns the number of visible lines in the drop down mode.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getDropDownLineCount() 

    def setListBoxDropDownLineCount( self, cCtrlName, nNumLines ): 
        """Sets the number of visible lines in the drop down mode.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setDropDownLineCount( nNumLines ) 

    def makeListBoxItemVisible( self, nItemPos ): 
        """Makes the item at the specified position visible by scrolling the list box.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.makeVisible( nItemPos ) 

    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlScrollBar
    #-------------------------------------------------- 

    def addScrollBar( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        bOrientation=1, 
                        itemListenerProc=None, 
                        actionListenerProc=None,
                        adjustmentListenerProc=None ): 
        """Add a ScrollBar control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlScrollBarModel" ) 
        oControlModel.Orientation = bOrientation
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if itemListenerProc != None: 
            self.addItemListenerProc( cCtrlName, itemListenerProc ) 
        if actionListenerProc != None: 
            self.addActionListenerProc( cCtrlName, actionListenerProc ) 
        if adjustmentListenerProc != None: 
            self.addAdjustmentListenerProc( cCtrlName, adjustmentListenerProc ) 

    def getScrollBarScrollValueMax( self, cCtrlName ): 
        """Get the number of items in a ListBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getMaximum() 

    def getScrollBarScrollValue( self, cCtrlName ): 
        """Return a tupple of all items in the ListBox.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getValue()

    def setScrollBarScrollValueMax( self, cCtrlName, bValue ): 
        """Selects/Deselects the ispecified item.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setMaximum(bValue)

    def setScrollBarScrollValue( self, cCtrlName, bValue ): 
        """Returns the number of visible lines in the drop down mode.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setValue(bValue)

    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlProgressBar
    #-------------------------------------------------- 

    def addProgressBar( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        nBorder=1 ): 
        """Add an ProgressBar control to the window. 
        nBorder = 0:none, 1:3D border, 2:simple border""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlProgressBarModel" ) 
        oControlModel.Border = nBorder
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

    def getProgressValue( self, cCtrlName ): 
        """Return the value.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getValue()

    def setProgressRange( self, cCtrlName, bValueMin, bValueMax ): 
        """Sets the ispecified item.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setRange(bValueMin,bValueMax)

    def setProgressValue( self, cCtrlName, bValue ): 
        """Sets the number of visible lines in the drop down mode.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setValue(bValue)

        
    #-------------------------------------------------- 
    #   com.sun.star.awt.UnoControlGroupBox
    #-------------------------------------------------- 

    def addGroupBox( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight, 
                        cLabel=None ): 
        """Add a Group Box control to the window.""" 
        oControlModel = createUnoService( "com.sun.star.awt.UnoControlGroupBoxModel" ) 
        oControl      = createUnoService( oControlModel.DefaultControl ) # Create a control and its model. 
        oControl.setModel( oControlModel )                              # Introduce the model to the control. 
        oControl.createPeer( getAwtToolkit(), self.getWindow() )        # Make the control create its window peer. 
        # Keep a dictionary of controls we are managing in the window. 
        self.controls[ cCtrlName ] = oControl 
        # Set position and size 
        self.setPosSize( cCtrlName, nPositionX, nPositionY, nWidth, nHeight ) 

        if cLabel != None: 
            oControlModel.Label = cLabel 

    #-------------------------------------------------- 
    #   Access controls and control models 
    #-------------------------------------------------- 

    def getControl( self, cCtrlName ): 
        """Get the control (not its model) for a particular control name. 
        The control returned includes the service com.sun.star.awt.UnoControl, 
         and another control-specific service which inherits from it. 
        """ 
        oControl = self.controls[ cCtrlName ] 
        return oControl 


    def getControlModel( self, cCtrlName ): 
        """Get the control model (not the control) for a particular control name. 
        The model returned includes the service UnoControlModel, 
         and another control-specific service which inherits from it. 
        """ 
        oControl = self.getControl( cCtrlName ) 
        oControlModel = oControl.getModel() 
        return oControlModel 


    #-------------------------------------------------- 
    #   Adjust properties of control models 
    #-------------------------------------------------- 

    def setControlModelProperty( self, cCtrlName, cPropertyName, uValue ): 
        """Set the value of a property of a control's model. 
        This affects the control model, not the control. 
        """ 
        oControlModel = self.getControlModel( cCtrlName ) 
        oControlModel.setPropertyValue( cPropertyName, uValue ) 

    def getControlModelProperty( self, cCtrlName, cPropertyName ): 
        """Get the value of a property of a control's model. 
        This affects the control model, not the control. 
        """ 
        oControlModel = self.getControlModel( cCtrlName ) 
        return oControlModel.getPropertyValue( cPropertyName ) 
      

    #-------------------------------------------------- 
    #   Adjust controls (not models) 
    #-------------------------------------------------- 

    # The following apply to all controls which are a 
    #   com.sun.star.awt.UnoControl 
    # (which means all controls) 

    def setDesignMode( self, cCtrlName, bDesignMode=True ): 
        oControl = self.getControl( cCtrlName ) 
        oControl.setDesignMode( bDesignMode ) 

    def isDesignMode( self, cCtrlName, bDesignMode=True ): 
        oControl = self.getControl( cCtrlName ) 
        return oControl.isDesignMode() 
  
    def isTransparent( self, cCtrlName, bDesignMode=True ): 
        oControl = self.getControl( cCtrlName ) 
        return oControl.isTransparent() 

    def setPosSize( self, cCtrlName, nPositionX, nPositionY, nWidth, nHeight ): 
        """Set the position and size of the window.""" 
        oControl = self.getControl( cCtrlName ) 
        # if negative coordinates are given for X or Y position, 
        #  then make that coordinate be relative to the right/bottom 
        #  edge of the window instead of to the left/top. 
        if nPositionX < 0  or  nPositionY < 0: 
            oWindowPosSizeRect = self.getWindowPosSize() 
            if nPositionX < 0: nPositionX = oWindowPosSizeRect.Width  + nPositionX - nWidth 
            if nPositionY < 0: nPositionY = oWindowPosSizeRect.Height + nPositionY - nHeight 
        oControl.setPosSize( 1.2*nPositionX, 1.0*nPositionY, 1.2*nWidth, 1.3*nHeight, 
                                     uno.getConstantByName( "com.sun.star.awt.PosSize.POSSIZE" ) ) 
    def setPosition( self, cCtrlName, nPositionX, nPositionY ): 
        """Set the position of the window.""" 
        oControl = self.getControl( cCtrlName ) 
        # if negative coordinates are given for X or Y position, 
        #  then make that coordinate be relative to the right/bottom 
        #  edge of the window instead of to the left/top. 
        if nPositionX < 0  or  nPositionY < 0: 
            oCtrlPosSizeRect = oControl.getPosSize() 
            oWindowPosSizeRect = self.getWindowPosSize() 
            if nPositionX < 0: nPositionX = oWindowPosSizeRect.Width  + nPositionX - oCtrlPosSizeRect.Width 
            if nPositionY < 0: nPositionY = oWindowPosSizeRect.Height + nPositionY - oCtrlPosSizeRect.Height 
        oControl.setPosSize( nPositionX, nPositionY, 0, 0, 
                                     uno.getConstantByName( "com.sun.star.awt.PosSize.POS" ) ) 
    def setSize( self, cCtrlName, nWidth, nHeight ): 
        """Set the position and size of the window.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setPosSize( 0, 0, nWidth, nHeight, 
                                     uno.getConstantByName( "com.sun.star.awt.PosSize.SIZE" ) ) 
    def getPosSize( self, cCtrlName ): 
        """Returns a com.sun.star.awt.Rectangle that contains the position and size of this window.""" 
        oControl = self.getControl( cCtrlName ) 
        return oControl.getPosSize() 

    def setVisible( self, cCtrlName, bVisible ): 
        """Show or hide the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setVisible( bVisible ) 

    def setEnable( self, cCtrlName, bEnable ): 
        """Enable or disable the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setEnable( bEnable ) 

    def setFocus( self, cCtrlName ): 
        """Set focus to the control.""" 
        oControl = self.getControl( cCtrlName ) 
        oControl.setFocus() 
    

    #-------------------------------------------------- 
    #   Add listeners to controls. 
    #-------------------------------------------------- 

    # This applies to... 
    #   UnoControlButton 
    #   UnoControlComboBox 
    def addActionListenerProc( self, cCtrlName, actionListenerProc ): 
        """Create an com.sun.star.awt.XActionListener object and add it to a control. 
        A listener object is created which will call the python procedure actionListenerProc. 
        The actionListenerProc can be either a method or a global procedure. 
        The following controls support XActionListener: 
            UnoControlButton 
        """ 
        oControl = self.getControl( cCtrlName ) 
        oActionListener = ActionListenerProcAdapter( actionListenerProc ) 
        oControl.addActionListener( oActionListener ) 

    # This applies to... 
    #   UnoControlCheckBox 
    #   UnoControlComboBox 
    def addItemListenerProc( self, cCtrlName, itemListenerProc ): 
        """Create an com.sun.star.awt.XItemListener object and add it to a control. 
        A listener object is created which will call the python procedure itemListenerProc. 
        The itemListenerProc can be either a method or a global procedure. 
        The following controls support XActionListener: 
            UnoControlCheckBox 
        """ 
        oControl = self.getControl( cCtrlName ) 
        oActionListener = ItemListenerProcAdapter( itemListenerProc ) 
        oControl.addItemListener( oActionListener ) 

    # This applies to... 
    #   UnoControlEdit 
    def addTextListenerProc( self, cCtrlName, textListenerProc ): 
        """Create an com.sun.star.awt.XTextListener object and add it to a control. 
        A listener object is created which will call the python procedure textListenerProc. 
        The textListenerProc can be either a method or a global procedure. 
        The following controls support XTextListener: 
            UnoControlEdit 
        """ 
        oControl = self.getControl( cCtrlName ) 
        oTextListener = TextListenerProcAdapter( textListenerProc ) 
        oControl.addTextListener( oTextListener ) 
    
    # This applies to... 
    #   UnoScrollBar 
    def addAdjustmentListenerProc( self, cCtrlName, adjustmentListenerProc ): 
        """Create an com.sun.star.awt.XTextListener object and add it to a control. 
        A listener object is created which will call the python procedure textListenerProc. 
        The textListenerProc can be either a method or a global procedure. 
        The following controls support XTextListener: 
            UnoControlEdit 
        """ 
        oControl = self.getControl( cCtrlName ) 
        oAdjListener = AdjustmentListenerProcAdapter( adjustmentListenerProc ) 
        oControl.addAdjustmentListener( oAdjListener ) 

    # This applies to... 
    #   UnoScrollBar 
    def addKeyListenerProc( self, cCtrlName, keyListenerProc ): 
        """Create an com.sun.star.awt.XKeyListener object and add it to a control. 
        A listener object is created which will call the python procedure textListenerProc. 
        The textListenerProc can be either a method or a global procedure. 
        The following controls support XKeyListener: 
            UnoControlEdit 
        """ 
        oControl = self.getControl( cCtrlName ) 
        oKeyListener = KeyListenerProcAdapter( keyListenerProc ) 
        oControl.addKeyListener( oKeyListener ) 




from com.sun.star.awt import XTopWindowListener 
from com.sun.star.awt import XWindowListener 
from com.sun.star.awt import XFocusListener 
from com.sun.star.awt import XKeyListener 
from com.sun.star.awt import XMouseListener 
from com.sun.star.awt import XPaintListener 
from com.sun.star.lang import XEventListener 
class DBListenerWindow( DBWindow, unohelper.Base, 
                            XKeyListener, XMouseListener, XPaintListener, 
                            XFocusListener, XWindowListener, XTopWindowListener, 
                            XEventListener ): 
    """This is like a DBWindow, but has useful listener methods implemented directly in this class. 
    You can easily subclass this class, and override the various listener methods from the interfaces implemented by this class. 
    """ 
    def __init__( self, cTitle=None, tBoundsRect=None, nWindowAttributes=None ): 
        DBWindow.__init__( self, 
                    cTitle=cTitle, 
                    tBoundsRect=tBoundsRect, 
                    nWindowAttributes=nWindowAttributes ) 

        # Since this class implements the XTopWindowListener interface, 
        #  add ourself as a listener to our own window. 
        self.oWindow.addTopWindowListener( self ) 

        # Since this class implements the XWindowListener interface, 
        #  add ourself as a listener to our own window. 
        self.oWindow.addWindowListener( self ) 

        # Since this class implements the XFocusListener interface, 
        #  add ourself as a listener to our own window. 
        self.oWindow.addFocusListener( self ) 

        # Since this class implements the XKeyListener interface, 
        #  add ourself as a listener to our own window. 
        self.oWindow.addKeyListener( self ) 

        # Since this class implements the XMouseListener interface, 
        #  add ourself as a listener to our own window. 
        self.oWindow.addMouseListener( self ) 

        # Since this class implements the XPaintListener interface, 
        #  add ourself as a listener to our own window. 
        self.oWindow.addPaintListener( self ) 

    #-------------------------------------------------- 
    #   Interface:  XTopWindowListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.lang.EventObject has the following members: 
    #       Source  which is a  com.sun.star.uno.XInterface 
    #           refers to the object that fired the event. 
    #-------------------------------------------------- 

    # [oneway] void 
    # windowOpened( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowOpened( self, tEvent ): 
        """is invoked when a window has been opened.""" 
        pass 

    # [oneway] void 
    # windowClosing( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowClosing( self, tEvent ): 
        """is invoked when a window is in the process of being closed. 
        The close operation can be overridden at this point.""" 
        self.getWindow().dispose() 

    # [oneway] void 
    # windowClosed( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowClosed( self, tEvent ): 
        """is invoked when a window has been closed.""" 
        pass 

    # [oneway] void 
    # windowMinimized( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowMinimized( self, tEvent ): 
        """is invoked when a window is iconified.""" 
        pass 

    # [oneway] void 
    # windowNormalized( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowNormalized( self, tEvent ): 
        """is invoked when a window is de-iconified.""" 
        pass 

    # [oneway] void 
    # windowActivated( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowActivated( self, tEvent ): 
        """is invoked when a window is activated.""" 
        pass 

    # [oneway] void 
    # windowDeactivated( [in] com.sun.star.lang.EventObject tEvent ); 
    def windowDeactivated( self, tEvent ): 
        """is invoked when a window is de-activated.""" 
        pass 

    #-------------------------------------------------- 
    #   Interface:  XWindowListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.awt.WindowEvent has the following members: 
    #       long X      long Y 
    #       long Width  long Height 
    #       long LeftInset  long TopInset   long RightInset     long BottomInset 
    #-------------------------------------------------- 

    # void 
    # windowResized( [in] WindowEvent tEvent ); 
    def windowResized( self, tEvent ): 
        """is invoked when the window has been resized.""" 
        pass 

    # void 
    # windowMoved( [in] WindowEvent tEvent ); 
    def windowMoved( self, tEvent ): 
        """is invoked when the window has been moved.""" 
        pass 

    # void 
    # windowShown( [in] EventObject tEvent ); 
    def windowShown( self, tEvent ): 
        """is invoked when the window has been shown.""" 
        # please note the type of parameter is described 
        #  above in the comment for XTopWindowListener. 
        pass 

    # void 
    # windowHidden( [in] EventObject tEvent ); 
    def windowHidden( self, tEvent ): 
        """is invoked when the window has been hidden.""" 
        # please note the type of parameter is described 
        #  above in the comment for XTopWindowListener. 
        pass 

    #-------------------------------------------------- 
    #   Interface:  XFocusListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.awt.FocusEvent has the following members: 
    #       short FocusFlags 
    #       com.sun.star.uno.XInterface NextFocus 
    #       boolean Temporary 
    #-------------------------------------------------- 

    # [oneway] void 
    # focusGained( [in] FocusEvent tEvent ); 
    def focusGained( self, tEvent ): 
        """ is invoked when a window gains the keyboard focus.""" 
        pass 

    # [oneway] void 
    # focusLost( [in] FocusEvent tEvent ); 
    def focusLost( self, tEvent ): 
        """ is invoked when a window loses the keyboard focus.""" 
        pass 

    #-------------------------------------------------- 
    #   Interface:  XKeyListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.awt.KeyEvent has the following members: 
    #       short KeyCode   (constant from com.sun.star.awt.Key) 
    #       char  KeyChar 
    #       short KeyFunc   (constant from com.sun.star.awt.KeyFunction) 
    #-------------------------------------------------- 

    # [oneway] void 
    # keyPressed( [in] com.sun.star.awt.KeyEvent tEvent ); 
    def keyPressed( self, tEvent ): 
        """ is invoked when a key has been pressed.""" 
        pass 

    # [oneway] void 
    # keyReleased( [in] com.sun.star.awt.KeyEvent tEvent ); 
    def keyReleased( self, tEvent ): 
        """ is invoked when a key has been released."""
        pass 

    #-------------------------------------------------- 
    #   Interface:  XMouseListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.awt.MouseEvent has the following members: 
    #       short Buttons       (constant from com.sun.star.awt.MouseButton) 
    #       short X     short Y 
    #       long ClickCount 
    #       boolean PupupTrigger 
    #-------------------------------------------------- 

    # [oneway] void 
    # mousePressed( [in] com.sun.star.awt.MouseEvent tEvent ); 
    def mousePressed( self, tEvent ): 
        """is invoked when a mouse button has been pressed on a window.""" 
        pass 

    # [oneway] void 
    # mouseReleased( [in] com.sun.star.awt.MouseEvent tEvent ); 
    def mouseReleased( self, tEvent ): 
        """is invoked when a mouse button has been released on a window.""" 
        pass 

    # [oneway] void 
    # mouseEntered( [in] com.sun.star.awt.MouseEvent tEvent ); 
    def mouseEntered( self, tEvent ): 
        """is invoked when the mouse enters a window.""" 
        pass 

    # [oneway] void 
    # mouseExited( [in] com.sun.star.awt.MouseEvent tEvent ); 
    def mouseExited( self, tEvent ): 
        """is invoked when the mouse exits a window.""" 
        pass 

    #-------------------------------------------------- 
    #   Interface:  XPaintListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.awt.PaintEvent has the following members: 
    #       com.sun.star.awt.Rectangle UpdateRect 
    #       short Count 
    #-------------------------------------------------- 

    # [oneway] void 
    # windowPaint( [in] PaintEvent tEvent ); 
    def windowPaint( self, tEvent ): 
        """ is invoked when a region of the window became invalid, e.g. when another window has been moved away.""" 
        pass 

    #-------------------------------------------------- 
    #   Interface:  XEventListener 
    # 
    #   This interface is implemented so that your subclass 
    #    can conveniently override these methods! 
    # 
    #   Note that com.sun.star.lang.EventObject has the following members: 
    #       Source  which is a  com.sun.star.uno.XInterface 
    #           refers to the object that fired the event. 
    #-------------------------------------------------- 

    # void 
    # disposing( [in] com.sun.star.lang.EventObject tEvent ); 
    def disposing( self, tEvent ): 
        """gets called when the broadcaster is about to be disposed.""" 
        pass 
