# -*- coding: utf-8 -*-
#
#       config.py
#
#       Copyright 2012 David Klasinc <bigwhale@lubica.net>
#       Copyright 2010 Andrew <andrew@karmic-desktop>
#
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 3 of the License, or
#       (at your option) any later version.
#
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#       MA 02110-1301, USA.

import os
from ConfigParser import SafeConfigParser, NoSectionError, NoOptionError
from xdg.BaseDirectory import xdg_config_home

class KazamConfig(SafeConfigParser):

    DEFAULTS = [{
                "name": "main",
                "keys": {
                         "video_toggled":  "True",
                         "video_source":   "0",
                         "audio_toggled":  "False",
                         "audio_source":   "0",
                         "audio_volume":   "0",
                         "audio2_toggled": "False",
                         "audio2_source":  "0",
                         "audio2_volume":  "0",
                         "codec":          "0",
                         "counter":        "5",
                         "capture_cursor": "True",
                         "framerate":      "15",
                         },
                },
                {
                 "name": "keyboard_shortcuts",
                 "keys": {
                          "pause":  "<Shift><Control>p",
                          "finish": "<Shift><Control>f",
                          "show":   "<Shift><Control>s",
                          "quit":   "<Shift><Control>q",
                         },
                 }]

    CONFIGDIR = os.path.join(xdg_config_home, "kazam")
    CONFIGFILE = os.path.join(CONFIGDIR, "kazam.conf")

    def __init__(self):
        SafeConfigParser.__init__(self, self.DEFAULTS[0]['keys'])
        if not os.path.isdir(self.CONFIGDIR):
            os.makedirs(self.CONFIGDIR)
        if not os.path.isfile(self.CONFIGFILE):
            self.create_default()
            self.write()
        self.read(self.CONFIGFILE)

    def create_default(self):
        # For every section
        for section in self.DEFAULTS:
            # Add the section
            self.add_section(section["name"])
            # And add every key in it, with its default value
            for key in section["keys"]:
                value = section["keys"][key]
                self.set(section["name"], key, value)

    def find_default(self, section, key):
        for d_section in self.DEFAULTS:
            if d_section["name"] == section:
                for d_key in d_section["keys"]:
                    if d_key == key:
                        return d_section["keys"][key]

    def get(self, section, key):
        try:
            return SafeConfigParser.get(self, section, key)
        except NoSectionError:
            default = self.find_default(section, key)
            self.set(section, key, default)
            self.write()
            return default
        except NoOptionError:
            default = self.find_default(section, key)
            self.set(section, key, default)
            self.write()
            return default

    def set(self, section, option, value):
        # If the section referred to doesn't exist (rare case),
        # then create it
        if not self.has_section(section):
            self.add_section(section)
        SafeConfigParser.set(self, section, option, str(value))

    def write(self):
        file_ = open(self.CONFIGFILE, "w")
        SafeConfigParser.write(self, file_)
        file_.close()


