/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "edit_sql_script_wizard.h"
#include "grt/grt_manager.h"
#include "grtpp_util.h"
#include "base/string_utilities.h"
#include "sqlide/wb_sql_editor_form.h"

//----------------- SqlScriptSelectionPage ----------------------------------------------------------------

SqlScriptSelectionPage::SqlScriptSelectionPage(WizardForm* wizard)
  : WizardPage(wizard, "sql script selection page"), _file_selector(true), _context(NULL)
{
  set_short_title(_("Select SQL Script"));
  set_title(_("SQL File Selection"));

  // file selector
  _file_selector_heading.set_text(_("Please select SQL file you want to open."));
  _file_selector_heading.set_wrap_text(true);
  add(&_file_selector_heading, false, false);

  add(&_table, false, false);
  _table.set_row_count(1);
  _table.set_column_count(2);
  _table.set_column_spacing(4);

  _caption.set_text_align(mforms::WizardLabelAlignment);
  _caption.set_text(_("SQL script file:")); 
  _table.add(&_caption, 0, 1, 0, 1, mforms::HFillFlag);

  _file_selector.initialize("", mforms::OpenFile, "SQL Files (*.sql)|*.sql", _("Browse..."),
    false, boost::bind(&WizardPage::validate, this));
  scoped_connect(_file_selector.signal_changed(),boost::bind(&SqlScriptSelectionPage::file_changed, this)); 

  _table.add(&_file_selector, 1, 2, 0, 1, mforms::HExpandFlag | mforms::HFillFlag);

  // options
  _options_selector_heading.set_text(_("Options:"));
  _options_selector_heading.set_wrap_text(true);
  add(&_options_selector_heading, false, false);

  add(&_autorun_check, false, false);
  _autorun_check.set_text(_("Execute file after opening"));
  _autorun_check.set_active(false);
}

//--------------------------------------------------------------------------------------------------

void SqlScriptSelectionPage::file_changed()
{
  validate();
}

//--------------------------------------------------------------------------------------------------

void SqlScriptSelectionPage::do_validate()
{
  std::string name= _file_selector.get_filename();
  if (!name.empty())
  {
    if (!g_file_test(name.c_str(), (GFileTest)(G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR)))
    {
      _form->set_problem(_("Invalid path"));
      return;
    }
  }
  _form->clear_problem();
}

//--------------------------------------------------------------------------------------------------

bool SqlScriptSelectionPage::allow_next()
{
  return !_file_selector.get_filename().empty();
}

//--------------------------------------------------------------------------------------------------

bool SqlScriptSelectionPage::advance()
{
  db_mgmt_ConnectionRef conn = _db_conn->get_connection();
  if (conn.is_valid())
  {
    _context->show_status_text("Opening SQL Editor...");          
    boost::shared_ptr<Db_sql_editor> db_sql_editor(_context->add_new_query_window(conn));
    if (db_sql_editor)
    {
      db_sql_editor->open_file(_file_selector.get_filename());
      if (_autorun_check.get_active())
        db_sql_editor->run_sql();
    }
  }
  
  return true;
}

//--------------------------------------------------------------------------------------------------

std::string SqlScriptSelectionPage::next_button_caption()
{
  return _("Finish");
}

//--------------------------------------------------------------------------------------------------

void SqlScriptSelectionPage::set_db_connection(DbConnection *value)
{
  _db_conn= value;
}

//----------------- EditSqlScriptWizard ---------------------------------------------------------
void SqlScriptSelectionPage::set_context(wb::WBContext* context)
{
  _context= context;
}

//----------------- EditSqlScriptWizard ---------------------------------------------------------

EditSqlScriptWizard::EditSqlScriptWizard(wb::WBContext* context)
:
WizardForm(context->get_grt_manager()),
_context(context)
{
  _db_conn.init(_context->get_root()->rdbmsMgmt(), false);

  set_title(_("Edit SQL Script"));

  _db_conn_page= new ConnectionPage(this);
  _db_conn_page->set_db_connection(&_db_conn);
  add_page(manage(_db_conn_page));

  _sql_script_selection_page= new SqlScriptSelectionPage(this);
  _sql_script_selection_page->set_db_connection(&_db_conn);
  _sql_script_selection_page->set_context(_context);
  add_page(manage(_sql_script_selection_page));
}

//--------------------------------------------------------------------------------------------------

EditSqlScriptWizard::~EditSqlScriptWizard()
{
  // Pages are freed by the WizardForm ancestor.
}

//--------------------------------------------------------------------------------------------------
