/* ---------------------------------------------------------------------
 * 
 * Giada - Your Hardcore Loopmachine 
 * 
 * ---------------------------------------------------------------------
 * 
 * Copyright (C) 2010-2012 Giovanni A. Zuliani | Monocasual
 * 
 * This file is part of Giada - Your Hardcore Loopmachine.
 * 
 * Giada - Your Hardcore Loopmachine is free software: you can 
 * redistribute it and/or modify it under the terms of the GNU General 
 * Public License as published by the Free Software Foundation, either 
 * version 3 of the License, or (at your option) any later version.
 * 
 * Giada - Your Hardcore Loopmachine is distributed in the hope that it 
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied 
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Giada - Your Hardcore Loopmachine. If not, see 
 * <http://www.gnu.org/licenses/>.
 * 
 * ------------------------------------------------------------------ */




#include <math.h>
#include "mixer.h"
#include "init.h"
#include "wave.h"
#include "gui_utils.h"
#include "recorder.h"
#include "pluginHost.h"
#include "patch.h"
#include "conf.h"
#include "mixerHandler.h"
#include "channel.h"


extern Mixer 			 G_Mixer;
extern Patch		 	 G_Patch;
extern Conf				 G_Conf;
#ifdef WITH_VST
extern PluginHost  G_PluginHost;
#endif


Mixer::Mixer() 	{}
Mixer::~Mixer() {}


#define TICKSIZE 38


float Mixer::tock[TICKSIZE] = {
	 0.059033,  0.117240,  0.173807,  0.227943,  0.278890,  0.325936,
	 0.368423,  0.405755,  0.437413,  0.462951,  0.482013,  0.494333,
	 0.499738,  0.498153,  0.489598,  0.474195,  0.452159,  0.423798,
	 0.389509,  0.349771,  0.289883,  0.230617,  0.173194,  0.118739,
	 0.068260,  0.022631, -0.017423, -0.051339,	-0.078721, -0.099345,
	-0.113163, -0.120295, -0.121028, -0.115804, -0.105209, -0.089954,
	-0.070862, -0.048844
};


float Mixer::tick[TICKSIZE] = {
	  0.175860,  0.341914,  0.488904,  0.608633,  0.694426,  0.741500,
	  0.747229,  0.711293,	0.635697,  0.524656,  0.384362,  0.222636,
	  0.048496, -0.128348, -0.298035, -0.451105, -0.579021, -0.674653,
	 -0.732667, -0.749830, -0.688924, -0.594091, -0.474481, -0.340160,
	 -0.201360, -0.067752,  0.052194,  0.151746,  0.226280,  0.273493,
	  0.293425,  0.288307,  0.262252,  0.220811,  0.170435,  0.117887,
	  0.069639,  0.031320
};





void Mixer::init() {
	quanto      = 1;
	docross     = false;
	rewindWait  = false;
	running     = false;
	ready       = true;
	waitRec     = 0;
	actualFrame = 0;
	bpm 		    = DEFAULT_BPM;
	bars		    = DEFAULT_BARS;
	beats		    = DEFAULT_BEATS;
	quantize    = DEFAULT_QUANTIZE;
	metronome   = false;

	tickTracker = 0;
	tockTracker = 0;
	tickPlay    = false;
	tockPlay    = false;

	outVol       = DEFAULT_OUT_VOL;
	inVol        = DEFAULT_IN_VOL;
	peakOut      = 0.0f;
	peakIn	     = 0.0f;
	chanInput    = NULL;
	inputTracker = 0;

	actualBeat   = 0;

	

	vChanInput   = NULL;
	vChanInToOut = (float *) malloc(kernelAudio::realBufsize * 2 * sizeof(float));

	pthread_mutex_init(&mutex_recs, NULL);
	pthread_mutex_init(&mutex_chans, NULL);
	pthread_mutex_init(&mutex_plugins, NULL);

	updateFrameBars();
	rewind();
}





channel *Mixer::addChannel(char side, int type) {
	channel *ch = (channel*) malloc(sizeof(channel));
	if (!ch) {
		printf("[mixer] unable to alloc memory for channel struct\n");
		return NULL;
	}

	

	ch->vChan = (float *) malloc(kernelAudio::realBufsize * 2 * sizeof(float));
	if (!ch->vChan) {
		printf("[mixer] unable to alloc memory for this vChan\n");
		return NULL;
	}

	while (true) {
		int lockStatus = pthread_mutex_trylock(&mutex_chans);
		if (lockStatus == 0) {
			channels.add(ch);
			pthread_mutex_unlock(&mutex_chans);
			break;
		}
	}

	initChannel(ch);

	ch->index = getNewIndex();
	ch->side  = side;
	ch->type  = type;

	printf("[mixer] channel index=%d added, type=%d, total=%d\n", ch->index, ch->type, channels.size);

	return ch;
}





int Mixer::getNewIndex() {

	

	if (channels.size == 1)
		return 0;

	int index = 0;
	for (unsigned i=0; i<channels.size-1; i++) {
		if (channels.at(i)->index > index)
			index = channels.at(i)->index;
		}
	index += 1;
	return index;
}





void Mixer::pushChannel(Wave *w, channel *ch) {
	ch->wave = w;
	ch->status = STATUS_OFF;
	ch->start     = 0;
	ch->startTrue = 0;
	ch->end       = ch->wave->size;
	ch->endTrue   = ch->wave->size;
}





void Mixer::initChannel(channel *ch) {
	ch->wave        = NULL;
	ch->key         = 0;
	ch->tracker     = 0;
	ch->status      = STATUS_EMPTY;
	ch->start       = 0;
	ch->startTrue   = 0;
	ch->end         = 0;
	ch->endTrue     = 0;
	ch->solo        = false;
	ch->mute        = false;
	ch->mute_i      = false;
	ch->mute_s      = false;
	ch->mode        = DEFAULT_CHANMODE;
	ch->volume      = DEFAULT_VOL;
	ch->volume_i    = 1.0f;
	ch->volume_d    = 0.0f;
	ch->pitch       = gDEFAULT_PITCH;
	ch->boost       = 1.0f;
	ch->panLeft     = 1.0f;
	ch->panRight    = 1.0f;
	ch->qWait	      = false;
	ch->recStatus   = REC_STOPPED;
	ch->fadein      = 1.0f;
	ch->fadeoutOn   = false;
	ch->fadeoutVol  = 1.0f;
	ch->fadeoutStep = DEFAULT_FADEOUT_STEP;

	ch->readActions = false;
	ch->hasActions  = false;

	
	

	gVector <class Plugin *> p;
	ch->plugins = p;

#ifdef WITH_VST 
	G_PluginHost.freeVstMidiEvents(ch, true);
#endif
}





int Mixer::deleteChannel(channel *ch) {
	int lockStatus;
	while (true) {
		lockStatus = pthread_mutex_trylock(&mutex_chans);
		if (lockStatus == 0) {
			ch->status = STATUS_OFF;
			int i = getChannelIndex(ch);
			if (ch->wave) {
				delete ch->wave;
				ch->wave = NULL;
			}
			free(ch->vChan);
			free(ch);
			channels.del(i);
			pthread_mutex_unlock(&mutex_chans);
			return 1;
		}
		
		
	}
}





void Mixer::freeChannel(channel *ch) {
	ch->status = STATUS_OFF;
	if (ch->wave != NULL) {
		delete ch->wave;
		ch->wave   = NULL;
	}
	ch->status = STATUS_EMPTY;
}






void Mixer::chanStop(channel *ch) {
	ch->status = STATUS_OFF;
	chanReset(ch);
}





int Mixer::getChannelIndex(channel *ch) {
	for (unsigned i=0; i<channels.size; i++)
		if (channels.at(i)->index == ch->index)
			return i;
	return -1;
}





channel *Mixer::getChannelByIndex(int index) {
	for (unsigned i=0; i<channels.size; i++)
		if (channels.at(i)->index == index)
			return channels.at(i);
	printf("[mixer::getChannelByIndex] channel at index %d not found!\n", index);
	return NULL;
}





void Mixer::chanReset(channel *ch)	{
	ch->tracker = ch->start;
	ch->mute_i  = false;
}





void Mixer::fadein(channel *ch, bool internal) {

	

	if (internal) ch->mute_i = false;
	else          ch->mute   = false;
	ch->fadein = 0.0f;
}





void Mixer::fadeout(channel *ch, int actionPostFadeout) {
	calcFadeoutStep(ch);
	ch->fadeoutOn   = true;
	ch->fadeoutVol  = 1.0f;
	ch->fadeoutType = FADEOUT;
	ch->fadeoutEnd	 = actionPostFadeout;
}





void Mixer::xfade(channel *ch) {
	calcFadeoutStep(ch);
	ch->fadeoutOn      = true;
	ch->fadeoutVol     = 1.0f;
	ch->fadeoutTracker = ch->tracker;
	ch->fadeoutType    = XFADE;
	chanReset(ch);
}





int Mixer::getChanPos(channel *ch)	{
	if (ch->status & ~(STATUS_EMPTY | STATUS_MISSING | STATUS_OFF))  
		return ch->tracker - ch->start;
	else
		return -1;
}





int Mixer::masterPlay(
	void *out_buf, void *in_buf, unsigned n_frames,
	double streamTime, RtAudioStreamStatus status, void *userData) {
	return G_Mixer.__masterPlay(out_buf, in_buf, n_frames);
}





int Mixer::__masterPlay(void *out_buf, void *in_buf, unsigned bufferFrames) {

	if (!ready)
		return 0;

	float *buffer  = ((float *) out_buf);
	float *inBuf   = ((float *) in_buf);
	bufferFrames  *= 2;     
	peakOut        = 0.0f;  
	peakIn         = 0.0f;  

	

	memset(buffer, 0, sizeof(float) * bufferFrames);         
	memset(vChanInToOut, 0, sizeof(float) * bufferFrames);   

	pthread_mutex_lock(&mutex_chans);
	for (unsigned i=0; i<channels.size; i++)
		memset(channels.at(i)->vChan, 0, sizeof(float) * bufferFrames);     
	pthread_mutex_unlock(&mutex_chans);

	for (unsigned j=0; j<bufferFrames; j+=2) {

		if (kernelAudio::inputEnabled) {

			

			if (inBuf[j] * inVol > peakIn)
				peakIn = inBuf[j] * inVol;

			

			if (inToOut) {
				vChanInToOut[j]   = inBuf[j]   * inVol;
				vChanInToOut[j+1] = inBuf[j+1] * inVol;
			}
		}

		

		if (running) {

			

			if (chanInput != NULL && kernelAudio::inputEnabled) {

				

				if (waitRec < G_Conf.delayComp)
					waitRec += 2;
				else {
					vChanInput[inputTracker]   += inBuf[j]   * inVol;
					vChanInput[inputTracker+1] += inBuf[j+1] * inVol;
					inputTracker += 2;
					if (inputTracker >= totalFrames)
						inputTracker = 0;
				}
			}

			

			if (quantize > 0 && quanto > 0) {

				bool isQuanto = actualFrame % (quanto) == 0 ? true : false;

				

				if (isQuanto && rewindWait) {
					rewindWait = false;
					rewind();
				}

				pthread_mutex_lock(&mutex_chans);
				for (unsigned k=0; k<channels.size; k++) {
					channel *ch = channels.at(k);
					if (ch->wave == NULL)
						continue;
					if (isQuanto && (ch->mode & SINGLE_ANY) && ch->qWait == true)	{

						

						if (ch->status == STATUS_OFF) {
							ch->status = STATUS_PLAY;
							ch->qWait  = false;
						}
						else
							xfade(ch);

						

						if (recorder::canRec(ch)) {
							if (ch->mode == SINGLE_PRESS)
								recorder::startOverdub(k, ACTION_KEYS, actualFrame);
							else
								recorder::rec(k, ACTION_KEYPRESS, actualFrame);
						}
					}
				}
				pthread_mutex_unlock(&mutex_chans);
			}

			

			if (actualFrame % framesPerBar == 0 && actualFrame != 0) {
				if (metronome)
					tickPlay = true;

				pthread_mutex_lock(&mutex_chans);
				for (unsigned k=0; k<channels.size; k++) {
					channel *ch = channels.at(k);
					if (ch->wave == NULL)
						continue;
					if (ch->mode == LOOP_REPEAT && ch->status == STATUS_PLAY)
						xfade(ch);
				}
				pthread_mutex_unlock(&mutex_chans);
			}


			

			if (actualFrame == 0) {
				pthread_mutex_lock(&mutex_chans);
				for (unsigned k=0; k<channels.size; k++) {
					channel *ch = channels.at(k);

					if (ch->wave == NULL)
						continue;

					if (ch->mode & (LOOP_ONCE | LOOP_BASIC | LOOP_REPEAT)) { 

						

						if (ch->status == STATUS_PLAY) {
							if (ch->mute || ch->mute_i)
								chanReset(ch);
							else
								xfade(ch);
						}
						else
						if (ch->status == STATUS_ENDING)
							chanStop(ch);
					}

					if (ch->status== STATUS_WAIT)
						ch->status = STATUS_PLAY;

					if (ch->recStatus == REC_ENDING) {
						ch->recStatus = REC_STOPPED;
						recorder::disableRead(ch);    
					}
					else
					if (ch->recStatus == REC_WAITING) {
						ch->recStatus = REC_READING;
						recorder::enableRead(ch);     
					}
				}
				pthread_mutex_unlock(&mutex_chans);
			}

			

			pthread_mutex_lock(&mutex_recs);
			for (unsigned y=0; y<recorder::frames.size; y++) {
				if (recorder::frames.at(y) == actualFrame) {
					for (unsigned z=0; z<recorder::global.at(y).size; z++) {
						int index   = recorder::global.at(y).at(z)->chan;
						channel *ch = getChannelByIndex(index);
						if (ch->readActions == false)
							continue;
						recorder::action *a = recorder::global.at(y).at(z);
						switch (a->type) {
							case ACTION_KEYPRESS:
								if (ch->mode & SINGLE_ANY) {
									mh_startChan(ch, false);
									break;
								}
							case ACTION_KEYREL:
								if (ch->mode & SINGLE_ANY) {
									mh_stopChan(ch);
									break;
								}
							case ACTION_KILLCHAN:
								if (ch->mode & SINGLE_ANY) {
									mh_killChan(ch);
									break;
								}
							case ACTION_MIDI:
#ifdef WITH_VST
								
								G_PluginHost.addVstMidiEvent(a->event, ch);
#endif
								break;
							case ACTION_MUTEON:
								mh_muteChan(ch, true);   
								break;
							case ACTION_MUTEOFF:
								mh_unmuteChan(ch, true); 
								break;
							case ACTION_VOLUME:
								calcVolumeEnv(ch, actualFrame);
								break;
						}
					}
					break;
				}
			}
			pthread_mutex_unlock(&mutex_recs);

			actualFrame += 2;

			

			if (actualFrame > totalFrames) {
				actualFrame = 0;
				actualBeat  = 0;
			}
			else if (actualFrame % framesPerBeat == 0 && actualFrame > 0) {
				actualBeat++;

				

				if (metronome && !tickPlay)
					tockPlay = true;
			}
		} 
	}

	

#ifdef WITH_VST
	pthread_mutex_lock(&mutex_chans);
	for (unsigned k=0; k<channels.size; k++) {
		channel *ch = channels.at(k);
		if (ch->type == CHANNEL_MIDI)
			processPlugins(ch);
	}
	pthread_mutex_unlock(&mutex_chans);
#endif

	

	for (unsigned j=0; j<bufferFrames; j+=2) {

		pthread_mutex_lock(&mutex_chans);
		for (unsigned k=0; k<channels.size; k++) {
			channel *ch = channels.at(k);

			

			if (ch->wave == NULL)
				continue;

			if (ch->status & (STATUS_PLAY | STATUS_ENDING)) {

				if (ch->tracker <= channels.at(k)->end) {

					

					unsigned ctp = ch->tracker * ch->pitch;

					

					if (ch->fadein <= 1.0f)
						ch->fadein += 0.01f;		

					

					if (running) {
						ch->volume_i += ch->volume_d;
						if (ch->volume_i < 0.0f)
							ch->volume_i = 0.0f;
						else
						if (ch->volume_i > 1.0f)
							ch->volume_i = 1.0f;
					}

					

					if (ch->fadeoutOn) {
						if (ch->fadeoutVol >= 0.0f) { 

							float v = ch->volume_i * ch->boost;

							if (ch->fadeoutType == XFADE) {

								

								unsigned ftp = ch->fadeoutTracker * ch->pitch;

								ch->vChan[j]   += ch->wave->data[ftp]   * ch->fadeoutVol * ch->panLeft  * v;
								ch->vChan[j+1] += ch->wave->data[ftp+1] * ch->fadeoutVol * ch->panRight * v;

								ch->vChan[j]   += ch->wave->data[ctp]   * ch->panLeft  * v;
								ch->vChan[j+1] += ch->wave->data[ctp+1] * ch->panRight * v;

							}
							else { 
								ch->vChan[j]   += ch->wave->data[ctp]   * ch->fadeoutVol * ch->panLeft  * v;
								ch->vChan[j+1] += ch->wave->data[ctp+1] * ch->fadeoutVol * ch->panRight * v;
							}

							ch->fadeoutVol     -= ch->fadeoutStep;
							ch->fadeoutTracker += 2;
						}
						else {  
							ch->fadeoutOn  = false;
							ch->fadeoutVol = 1.0f;

							

							if (ch->fadeoutType == XFADE) {
								ch->qWait = false;
							}
							else {
								if (ch->fadeoutEnd == DO_MUTE)
									ch->mute = true;
								else
								if (ch->fadeoutEnd == DO_MUTE_I)
									ch->mute_i = true;
								else              
									chanStop(ch);
							}

							

							ch->vChan[j]   = ch->vChan[j-2];
							ch->vChan[j+1] = ch->vChan[j-1];
						}
					}  
					else {
						if (!ch->mute && !ch->mute_i) {
							float v = ch->volume_i * ch->fadein * ch->boost;
							ch->vChan[j]   += ch->wave->data[ctp]   * v * ch->panLeft;
							ch->vChan[j+1] += ch->wave->data[ctp+1] * v * ch->panRight;
						}
					}

					ch->tracker += 2;

					

					if (ch->tracker >= ch->end) {

						chanReset(ch);

						if (ch->mode & (SINGLE_BASIC | SINGLE_PRESS | SINGLE_RETRIG) ||
						   (ch->mode == SINGLE_ENDLESS && ch->status == STATUS_ENDING))
						{
							ch->status = STATUS_OFF;
						}

						
						

						if ((ch->mode & LOOP_ANY) && !running)
							ch->status = STATUS_OFF;

						

						if (ch->mode == LOOP_ONCE && ch->status != STATUS_ENDING)
							ch->status = STATUS_WAIT;
					}
				}
			}
		} 
		pthread_mutex_unlock(&mutex_chans);

		

		if (tockPlay) {
			buffer[j]   += tock[tockTracker];
			buffer[j+1] += tock[tockTracker];
			tockTracker++;
			if (tockTracker >= TICKSIZE-1) {
				tockPlay    = false;
				tockTracker = 0;
			}
		}
		if (tickPlay) {
			buffer[j]   += tick[tickTracker];
			buffer[j+1] += tick[tickTracker];
			tickTracker++;
			if (tickTracker >= TICKSIZE-1) {
				tickPlay    = false;
				tickTracker = 0;
			}
		}
	} 

	

	pthread_mutex_lock(&mutex_chans);
	for (unsigned k=0; k<channels.size; k++) {
		channel *ch = channels.at(k);
#ifdef WITH_VST
		if (ch->type == CHANNEL_SAMPLE)
			processPlugins(ch);
#endif
		for (unsigned j=0; j<bufferFrames; j+=2) {
			buffer[j]   += ch->vChan[j]   * ch->volume;
			buffer[j+1] += ch->vChan[j+1] * ch->volume;
		}
	}
	pthread_mutex_unlock(&mutex_chans);

	

#ifdef WITH_VST
	pthread_mutex_lock(&mutex_plugins);
	G_PluginHost.processStack(buffer, PluginHost::MASTER_OUT);
	G_PluginHost.processStack(vChanInToOut, PluginHost::MASTER_IN);
	pthread_mutex_unlock(&mutex_plugins);
#endif

	

	for (unsigned j=0; j<bufferFrames; j+=2) {

		

		if (inToOut) {
			buffer[j]   += vChanInToOut[j];
			buffer[j+1] += vChanInToOut[j+1];
		}

		buffer[j]   *= outVol;
		buffer[j+1] *= outVol;

		

		if (buffer[j] > peakOut)
			peakOut = buffer[j];

		if (G_Conf.limitOutput) {
			if (buffer[j] > 1.0f)
				buffer[j] = 1.0f;
			else if (buffer[j] < -1.0f)
				buffer[j] = -1.0f;

			if (buffer[j+1] > 1.0f)
				buffer[j+1] = 1.0f;
			else if (buffer[j+1] < -1.0f)
				buffer[j+1] = -1.0f;
		}
	}

	return 0;
}





void Mixer::updateFrameBars() {

	

	float seconds   = (60.0f / bpm) * beats;
	totalFrames     = G_Conf.samplerate * seconds * 2;
	framesPerBar    = totalFrames / bars;
	framesPerBeat   = totalFrames / beats;

	

	if (totalFrames % 2 != 0)
		totalFrames--;
	if (framesPerBar % 2 != 0)
		framesPerBar--;
	if (framesPerBeat % 2 != 0)
		framesPerBeat--;

	updateQuanto();

	

	if (vChanInput != NULL)
		free(vChanInput);
	vChanInput = (float*) malloc(totalFrames * sizeof(float));
	if (!vChanInput)
		printf("[Mixer] vChanInput realloc error!");
}





int Mixer::close() {
	
	running = false;
	while (channels.size > 0)
		deleteChannel(channels.at(0));
	
	free(vChanInput);
	free(vChanInToOut);
	return 1;
}





bool Mixer::isSilent() {
	for (unsigned i=0; i<channels.size; i++)
		if (channels.at(i)->status == STATUS_PLAY)
			return false;
	return true;
}





void Mixer::rewind() {

	actualFrame = 0;
	actualBeat  = 0;

	if (running)
		for (unsigned i=0; i<channels.size; i++) {
			channel *ch = channels.at(i);
			if (ch->wave != NULL) {

				

				if ((ch->mode & LOOP_ANY) || (ch->recStatus == REC_READING && (ch->mode & SINGLE_ANY)))
					chanReset(ch);
			}
		}
}





void Mixer::updateQuanto() {

	

	if (quantize != 0)
		quanto = framesPerBeat / quantize;
	if (quanto % 2 != 0)
		quanto++;
}





void Mixer::calcFadeoutStep(channel *ch) {

	

	unsigned ctracker = ch->tracker * ch->pitch;

	if (ch->end - ctracker < (1 / DEFAULT_FADEOUT_STEP) * 2)
		ch->fadeoutStep = ceil((ch->end - ctracker) / ch->volume) * 2; 
	else
		ch->fadeoutStep = DEFAULT_FADEOUT_STEP;
}





bool Mixer::hasLogicalSamples() {
	for (unsigned i=0; i<channels.size; i++)
		if (channels.at(i)->wave != NULL)
			if (channels.at(i)->wave->isLogical)
				return true;
	return false;
}





bool Mixer::hasEditedSamples() {
	for (unsigned i=0; i<channels.size; i++)
		if (channels.at(i)->wave != NULL)
			if (channels.at(i)->wave->isEdited)
				return true;
	return false;
}





void Mixer::setPitch(channel *ch, float val) {

	

	if (val == 1.0f) {
		ch->start = ch->startTrue;
		ch->end   = ch->endTrue;
		ch->pitch = 1.0f;
		return;
	}

	ch->start = (unsigned) floorf(ch->startTrue / val);
	ch->end   = (unsigned) floorf(ch->endTrue   / val);

	ch->pitch = val;

	

	if (ch->start % 2 != 0)	ch->start++;
	if (ch->end   % 2 != 0)	ch->end++;

	

	if (ch->tracker > ch->end)
		ch->tracker = ch->end;
}





void Mixer::setChanStart(struct channel *ch, unsigned val) {
	if (val % 2 != 0)
		val++;
	ch->startTrue = val;
	ch->start     = (unsigned) floorf(ch->startTrue / ch->pitch);
	ch->tracker   = ch->start;
}





void Mixer::setChanEnd(struct channel *ch, unsigned val) {
	if (val % 2 != 0)
		val++;
	ch->endTrue = val;
	ch->end = (unsigned) floorf(ch->endTrue / ch->pitch);
}





bool Mixer::mergeVirtualInput() {
	if (vChanInput == NULL) {
		puts("[Mixer] virtual input channel not alloc'd");
		return false;
	}
	else {
#ifdef WITH_VST
		G_PluginHost.processStackOffline(vChanInput, PluginHost::MASTER_IN, 0, totalFrames);
#endif
		int numFrames = totalFrames*sizeof(float);
		memcpy(chanInput->wave->data, vChanInput, numFrames);
		memset(vChanInput, 0, numFrames); 
		return true;
	}
}





bool Mixer::isPlaying(channel *ch) {
	return ch->status == STATUS_PLAY || ch->status == STATUS_ENDING;
}





#ifdef WITH_VST
void Mixer::processPlugins(channel *ch) {
	pthread_mutex_lock(&mutex_plugins);
	G_PluginHost.processStack(ch->vChan, PluginHost::CHANNEL, ch);
	G_PluginHost.freeVstMidiEvents(ch);
	pthread_mutex_unlock(&mutex_plugins);
}
#endif





void Mixer::calcVolumeEnv(struct channel *ch, int frame) {

	

	recorder::action *a0 = NULL;
	recorder::action *a1 = NULL;
	int res;

	res = recorder::getAction(ch->index, ACTION_VOLUME, frame, &a0);
	if (res == 0) {
		
		return;
	}

	

	res = recorder::getNextAction(ch->index, ACTION_VOLUME, frame, &a1);

	

	if (res == -1) {
		
		res = recorder::getAction(ch->index, ACTION_VOLUME, 0, &a1);
	}
	else

	

	
	
	
	

	ch->volume_i = a0->fValue;
	ch->volume_d = ((a1->fValue - a0->fValue) / ((a1->frame - a0->frame) / 2)) * 1.003f;
}
