/* ---------------------------------------------------------------------
 * 
 * Giada - Your Hardcore Loopmachine 
 * 
 * ---------------------------------------------------------------------
 * 
 * Copyright (C) 2010-2012 Giovanni A. Zuliani | Monocasual
 * 
 * This file is part of Giada - Your Hardcore Loopmachine.
 * 
 * Giada - Your Hardcore Loopmachine is free software: you can 
 * redistribute it and/or modify it under the terms of the GNU General 
 * Public License as published by the Free Software Foundation, either 
 * version 3 of the License, or (at your option) any later version.
 * 
 * Giada - Your Hardcore Loopmachine is distributed in the hope that it 
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied 
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Giada - Your Hardcore Loopmachine. If not, see 
 * <http://www.gnu.org/licenses/>.
 * 
 * ------------------------------------------------------------------ */




#ifdef WITH_VST


#include "pluginHost.h"
#include "conf.h"
#include "const.h"
#include "mixer.h"
#include "gd_mainWindow.h"
#include "channel.h"


extern Conf          G_Conf;
extern Mixer         G_Mixer;
extern PluginHost    G_PluginHost;
extern unsigned      G_beats;
extern gdMainWindow *mainWin;


PluginHost::PluginHost() {

	

	vstTimeInfo.samplePos          = 0.0;
	vstTimeInfo.sampleRate         = G_Conf.samplerate;
	vstTimeInfo.nanoSeconds        = 0.0;
	vstTimeInfo.ppqPos             = 0.0;
	vstTimeInfo.tempo              = 120.0;
	vstTimeInfo.barStartPos        = 0.0;
	vstTimeInfo.cycleStartPos      = 0.0;
	vstTimeInfo.cycleEndPos        = 0.0;
	vstTimeInfo.timeSigNumerator   = 4;
	vstTimeInfo.timeSigDenominator = 4;
	vstTimeInfo.smpteOffset        = 0;
	vstTimeInfo.smpteFrameRate     = 1;
	vstTimeInfo.samplesToNextClock = 0;
	vstTimeInfo.flags              = 0;
}





PluginHost::~PluginHost() {}





int PluginHost::allocBuffers() {

	

	

	int bufSize = kernelAudio::realBufsize*sizeof(float);

	bufferI    = (float **) malloc(2 * sizeof(float*));
	bufferI[0] =  (float *) malloc(bufSize);
	bufferI[1] =  (float *) malloc(bufSize);

	bufferO    = (float **) malloc(2 * sizeof(float*));
	bufferO[0] =  (float *) malloc(bufSize);
	bufferO[1] =  (float *) malloc(bufSize);

	memset(bufferI[0], 0, bufSize);
	memset(bufferI[1], 0, bufSize);
	memset(bufferO[0], 0, bufSize);
	memset(bufferO[1], 0, bufSize);

	printf("[pluginHost] buffers allocated, buffersize = %d\n", 2*kernelAudio::realBufsize);

	

	return 1;
}





VstIntPtr VSTCALLBACK PluginHost::HostCallback(AEffect *effect, VstInt32 opcode, VstInt32 index, VstIntPtr value, void *ptr, float opt) {
	return G_PluginHost.gHostCallback(effect, opcode, index, value, ptr, opt);
}





VstIntPtr PluginHost::gHostCallback(AEffect *effect, VstInt32 opcode, VstInt32 index, VstIntPtr value, void *ptr, float opt) {

	

	switch (opcode) {

		

		case audioMasterAutomate:
			return 0;

		

		case audioMasterVersion:
			return kVstVersion;

		

		case audioMasterIdle:
			return 0;

		

		case DECLARE_VST_DEPRECATED(audioMasterWantMidi):
			return 0;

		

		case audioMasterGetTime:
			vstTimeInfo.samplePos          = G_Mixer.actualFrame;
			vstTimeInfo.sampleRate         = G_Conf.samplerate;
			vstTimeInfo.tempo              = G_Mixer.bpm;
			vstTimeInfo.timeSigNumerator   = G_Mixer.beats;
			vstTimeInfo.timeSigDenominator = G_Mixer.bars;
			vstTimeInfo.ppqPos             = (G_Mixer.actualFrame / (float) G_Conf.samplerate) * (float) G_Mixer.bpm / 60.0f;
			return (VstIntPtr) &vstTimeInfo;

		

		case audioMasterProcessEvents:
			return 0;

		

		case audioMasterIOChanged:
			return false;

		

		case DECLARE_VST_DEPRECATED(audioMasterNeedIdle):
			return 0;

		

		case audioMasterSizeWindow: {
			gWindow *window = NULL;
			for (unsigned i=0; i<masterOut.size && !window; i++)
				if (masterOut.at(i)->getPlugin() == effect)
					window = masterOut.at(i)->window;

			for (unsigned i=0; i<masterIn.size && !window; i++)
				if (masterIn.at(i)->getPlugin() == effect)
					window = masterIn.at(i)->window;

			for (unsigned i=0; i<G_Mixer.channels.size && !window; i++) {
				channel *ch = G_Mixer.channels.at(i);
				for (unsigned j=0; j<ch->plugins.size && !window; j++)
					if (ch->plugins.at(j)->getPlugin() == effect)
						window = ch->plugins.at(j)->window;
			}

			if (window) {
				printf("[pluginHost] audioMasterSizeWindow: resizing window from plugin %p\n", (void*) effect);
				if (index == 1 || value == 1)
					puts("[pluginHost] warning: non-sense values!");
				else
					window->size((int)index, (int)value);
				return 1;
			}
			else {
				printf("[pluginHost] audioMasterSizeWindow: window from plugin %p not found\n", (void*) effect);
				return 0;
			}
		}

		

		case audioMasterGetSampleRate:
			return G_Conf.samplerate;

		

		case audioMasterGetBlockSize:
			return kernelAudio::realBufsize;

		case audioMasterGetInputLatency:
			printf("[pluginHost] requested opcode 'audioMasterGetInputLatency' (%d)\n", opcode);
			return 0;

		case audioMasterGetOutputLatency:
			printf("[pluginHost] requested opcode 'audioMasterGetOutputLatency' (%d)\n", opcode);
			return 0;

		

		case audioMasterGetCurrentProcessLevel:
			return kVstProcessLevelRealtime;

		

		case audioMasterGetVendorString:
			strcpy((char*)ptr, "Monocasual");
			return 1;

		

		case audioMasterGetProductString:
			strcpy((char*)ptr, "Giada");
			return 1;

		

		case audioMasterGetVendorVersion:
			return (int) VERSIONE_FLOAT * 100;


		

		case audioMasterCanDo:
			printf("[pluginHost] audioMasterCanDo: %s\n", (char*)ptr);
			if (!strcmp((char*)ptr, "sizeWindow")       ||
					!strcmp((char*)ptr, "sendVstTimeInfo")  ||
					!strcmp((char*)ptr, "sendVstMidiEvent") ||
					!strcmp((char*)ptr, "sendVstMidiEventFlagIsRealtime"))
				return 1; 
			else
				return 0;

		

		case audioMasterUpdateDisplay:
			return 0;

		case audioMasterGetLanguage:
			return kVstLangEnglish;

		

		case audioMasterGetAutomationState:
			printf("[pluginHost] requested opcode 'audioMasterGetAutomationState' (%d)\n", opcode);
			return 0;

		

		case audioMasterBeginEdit:
			return 0;

		

		case audioMasterEndEdit:
			return 0;

		default:
			printf("[pluginHost] FIXME: host callback called with opcode %d\n", opcode);
			return 0;
	}
}





int PluginHost::addPlugin(const char *fname, int stackType, channel *ch) {

	Plugin *p    = new Plugin();
	bool success = true;

	gVector <Plugin *> *pStack;
	pStack = getStack(stackType, ch);

	if (!p->load(fname)) {
		
		
		success = false;
	}

	

	if (!success) {
		pStack->add(p);
		return 0;
	}

	

	else {

		

		if (!p->init(&PluginHost::HostCallback)) {
			delete p;
			return 0;
		}

		

		p->setup(G_Conf.samplerate, kernelAudio::realBufsize);

		

		int lockStatus;
		while (true) {
			lockStatus = pthread_mutex_trylock(&G_Mixer.mutex_plugins);
			if (lockStatus == 0) {
				pStack->add(p);
				pthread_mutex_unlock(&G_Mixer.mutex_plugins);
				break;
			}
		}

		char name[256]; p->getName(name);
		printf("[pluginHost] plugin id=%d loaded (%s), stack type=%d, stack size=%d\n", p->getId(), name, stackType, pStack->size);

		

		p->resume();

		return 1;
	}
}





void PluginHost::processStack(float *buffer, int stackType, channel *ch) {

	gVector <Plugin *> *pStack = getStack(stackType, ch);

	

	if (!G_Mixer.ready)
		return;
	if (pStack == NULL)
		return;
	if (pStack->size == 0)
		return;

	

	for (unsigned i=0; i<kernelAudio::realBufsize; i++) {
		bufferI[0][i] = buffer[i*2];
		bufferI[1][i] = buffer[(i*2)+1];
	}

	

	for (unsigned i=0; i<pStack->size; i++) {
		if (pStack->at(i)->status != 1)
			continue;
		if (pStack->at(i)->suspended)
			continue;
		if (pStack->at(i)->bypass)
			continue;
		if (ch) {   
			pStack->at(i)->processEvents((VstEvents*) &ch->events);
		}
		pStack->at(i)->processAudio(bufferI, bufferO, kernelAudio::realBufsize);
		bufferI = bufferO;
	}

	

	for (unsigned i=0; i<kernelAudio::realBufsize; i++) {
		buffer[i*2]     = bufferO[0][i];
		buffer[(i*2)+1] = bufferO[1][i];
	}
}





void PluginHost::processStackOffline(float *buffer, int stackType, channel *ch, int size) {

	

	

	int index = 0;
	int step  = kernelAudio::realBufsize*2;

	while (index <= size) {
		int left = index+step-size;
		if (left < 0)
			processStack(&buffer[index], stackType, ch);

	

		
		

		index+=step;
	}

}





Plugin *PluginHost::getPluginById(int id, int stackType, channel *ch) {
	gVector <Plugin *> *pStack = getStack(stackType, ch);
	for (unsigned i=0; i<pStack->size; i++) {
		if (pStack->at(i)->getId() == id)
			return pStack->at(i);
	}
	return NULL;
}





Plugin *PluginHost::getPluginByIndex(int index, int stackType, channel *ch) {
	gVector <Plugin *> *pStack = getStack(stackType, ch);
	if (pStack->size == 0)
		return NULL;
	if ((unsigned) index >= pStack->size)
		return NULL;
	return pStack->at(index);
}





void PluginHost::freeStack(int stackType, channel *ch) {

	gVector <Plugin *> *pStack;
	pStack = getStack(stackType, ch);

	if (pStack->size == 0)
		return;

	int lockStatus;
	while (true) {
		lockStatus = pthread_mutex_trylock(&G_Mixer.mutex_plugins);
		if (lockStatus == 0) {
			for (unsigned i=0; i<pStack->size; i++) {
				if (pStack->at(i)->status == 1) {  
					pStack->at(i)->suspend();
					pStack->at(i)->close();
				}
				delete pStack->at(i);
			}
			pStack->clear();
			pthread_mutex_unlock(&G_Mixer.mutex_plugins);
			break;
		}
	}

}





void PluginHost::freeAllStacks() {
	freeStack(PluginHost::MASTER_OUT);
	freeStack(PluginHost::MASTER_IN);
	for (unsigned i=0; i<G_Mixer.channels.size; i++)
		freeStack(PluginHost::CHANNEL, G_Mixer.channels.at(i));
}





void PluginHost::freePlugin(int id, int stackType, channel *ch) {

	gVector <Plugin *> *pStack;
	pStack = getStack(stackType, ch);

	

	for (unsigned i=0; i<pStack->size; i++)
		if (pStack->at(i)->getId() == id) {

			if (pStack->at(i)->status == 0) { 
				delete pStack->at(i);
				pStack->del(i);
				return;
			}
			else {
				int lockStatus;
				while (true) {
					lockStatus = pthread_mutex_trylock(&G_Mixer.mutex_plugins);
					if (lockStatus == 0) {
						pStack->at(i)->suspend();
						pStack->at(i)->close();
						delete pStack->at(i);
						pStack->del(i);
						pthread_mutex_unlock(&G_Mixer.mutex_plugins);
						printf("[pluginHost] plugin id=%d removed\n", id);
						return;
					}
					
						
				}
			}
		}
	printf("[pluginHost] plugin id=%d not found\n", id);
}





void PluginHost::swapPlugin(unsigned indexA, unsigned indexB, int stackType, channel *ch) {

	gVector <Plugin *> *pStack = getStack(stackType, ch);

	int lockStatus;
	while (true) {
		lockStatus = pthread_mutex_trylock(&G_Mixer.mutex_plugins);
		if (lockStatus == 0) {
			pStack->swap(indexA, indexB);
			pthread_mutex_unlock(&G_Mixer.mutex_plugins);
			printf("[pluginHost] plugin at index %d and %d swapped\n", indexA, indexB);
			return;
		}
		
			
	}
}





int PluginHost::getPluginIndex(int id, int stackType, channel *ch) {

	gVector <Plugin *> *pStack = getStack(stackType, ch);

	for (unsigned i=0; i<pStack->size; i++)
		if (pStack->at(i)->getId() == id)
			return i;
	return -1;
}





gVector <Plugin *> *PluginHost::getStack(int stackType, channel *ch) {
	switch(stackType) {
		case MASTER_OUT:
			return &masterOut;
		case MASTER_IN:
			return &masterIn;
		case CHANNEL:
			return &ch->plugins;
		default:
			return NULL;
	}
}





void PluginHost::addVstMidiEvent(VstMidiEvent *e, channel *ch) {
	if (ch->events.numEvents < MAX_VST_EVENTS) {
		ch->events.events[ch->events.numEvents] = (VstEvent*) e;
		ch->events.numEvents++;
		
	}
	else
		printf("[pluginHost] channel %d VstEvents = %d > MAX_VST_EVENTS, nothing to do\n", ch->index, ch->events.numEvents);
}





void PluginHost::addVstMidiEvent(uint32_t msg, channel *ch) {
	addVstMidiEvent(createVstMidiEvent(msg), ch);
}





VstMidiEvent *PluginHost::createVstMidiEvent(uint32_t msg) {

	VstMidiEvent *e = (VstMidiEvent*) malloc(sizeof(VstMidiEvent));

	

	e->type         = kVstMidiType;
	e->byteSize     = sizeof(VstMidiEvent);

	
	

	e->deltaFrames  = 0;

	

	e->flags        = kVstMidiEventIsRealtime;

	

	e->midiData[0]  = (msg >> 24) & 0xFF;  
	e->midiData[1]  = (msg >> 16) & 0xFF;  
	e->midiData[2]  = (msg >> 8)  & 0xFF;  
	e->midiData[3]  = 0;

	

	e->noteLength   = 0;

	

	e->noteOffset   = 0;

	

	e->noteOffVelocity = 0;

	return e;
}





void PluginHost::freeVstMidiEvents(channel *ch, bool init) {
	if (ch->events.numEvents == 0 && !init)
		return;
	memset(ch->events.events, 0, sizeof(VstEvent*) * MAX_VST_EVENTS);
	ch->events.numEvents = 0;
	ch->events.reserved  = 0;
	
}





unsigned PluginHost::countPlugins(int stackType, channel *ch) {
	gVector <Plugin *> *pStack = getStack(stackType, ch);
	return pStack->size;
}


#endif 
