# -*- coding: utf-8 -*-
"""
This file is part of Laborejo - http://www.laborejo.org
Author: Nils Gey info@laborejo.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

"""
In this file we have ultra high-level functions which to combine several
API features.
We can call these macros in here as well.
There is very little calculation and no need for any function parameters
just calling one api function after another.
"""

import laborejocore as api
import os, warnings, time
from laborejocore.items import Chord

class JackRunningError(Exception):
    pass

def calfboxInit(soundfont = None, autoconnect = True, midiInPort = None, clientName = "Laborejo"):
    if not soundfont:
        soundfont = os.path.join(os.path.dirname(__file__), "..", 'gm.sf2')
    try:
        import laborejocore.calfboxwrap as calfboxwrap
        api.cboxSmfCompatibleModule = calfboxwrap.Calfbox(384, soundfont, autoconnect, midiInPort, clientName) #Release the kraken!
        #We got this far, that means calfbox is now running. At least we have the classes/instances. Save it in core.Session.
        api._getSession().calfbox = api.cboxSmfCompatibleModule
    except ImportError:
       warnings.warn("calfbox not found. It is probably not installed as Python3 module. Playback broken. Not 'disabled', broken! You can use the rest of the program just fine.")
    except JackRunningError:
        warnings.warn("calfbox not initialised. Probably JACK is not running. Playback broken. Not 'disabled', broken! You can use the rest of the program just fine.")

def calfboxShutdown():
    if api.cboxSmfCompatibleModule: #maybe it was not running
        api.cboxSmfCompatibleModule.cbox.stop_audio()
        api.cboxSmfCompatibleModule.cbox.shutdown_engine()

    #else:
    #    pass #doesn't matter. the program will exit after that.

def jackMidiInToggleMute(boolean = None):
    if api.cboxSmfCompatibleModule:
        if not boolean is None and boolean: #but true and false
            api.cboxSmfCompatibleModule.jack_midi_in_is_open = True
        elif not boolean is None and not boolean:
            api.cboxSmfCompatibleModule.jack_midi_in_is_open = False
        else: #toggle
            api.cboxSmfCompatibleModule.jack_midi_in_is_open = not (api.cboxSmfCompatibleModule.jack_midi_in_is_open)

        api.l_send(lambda: api.l_global.jackMidiInToggleMute(api.cboxSmfCompatibleModule.jack_midi_in_is_open))
        return api.cboxSmfCompatibleModule.jack_midi_in_is_open #Allow midi in: True or False

def calfboxPlayChord(chordItem, force = False):
    """This is a slow feedback function. Do not use for sequencer-like
    realtime playback"""
    if api.cboxSmfCompatibleModule and (api.cboxSmfCompatibleModule.audibleFeedback or force) and type(chordItem) is api.items.Chord:
        api.cboxSmfCompatibleModule.sendChordEvent([api.pitch.toMidi(note.pitch) for note in chordItem.notelist]) #we cannot use this as a generator. We loop twice over it and also send the list to calfbox which needs it as a list.

def calfboxPlayCursorChord():
    calfboxPlayChord(api._getCursorItem(), force = True)

def calfboxGetPlaybackTicks():
    """Part of calfboxProcessor"""
    cboxTicks = api.cboxSmfCompatibleModule.transport.status().pos_ppqn
    laborejoTicks = cboxTicks + api.cboxSmfCompatibleModule.negativePlaybackTickOffset

    if api.cboxSmfCompatibleModule._SMF.cachedRepeatsOfLongestTrack and laborejoTicks >= api.cboxSmfCompatibleModule._SMF.cachedRepeatsOfLongestTrack[-1][0]: #the next repeat close
        fromTick, toTick = api.cboxSmfCompatibleModule._SMF.cachedRepeatsOfLongestTrack.pop() #get the next tuple. The list is reversed in order already so we can .pop()
        api.cboxSmfCompatibleModule.negativePlaybackTickOffset += toTick - fromTick

    api.l_score.playbackSetPosition(laborejoTicks) #THIS is the only exception of api.l_send. Because this whole function is wrapped in api.l_send by calfboxProcessor()

def calfboxSetTransportKeepRolling(boolean):
    if api.cboxSmfCompatibleModule:
        api.cboxSmfCompatibleModule.keepRolling = boolean

def calfboxProcessor():
    """Query calfbox for new midi data and react accordingly.
    Intended to use periodically in a GUI idle loop, event timer or so.
    It is expected to be called several times per second, so keep
    any changes brief and small"""
    #For performance reasons: No check if the calfbox module is running. Nobody should be that stupid and call that function in a script or manually.
    events = api.cboxSmfCompatibleModule.cbox.get_new_events()
    resetSound = api.cboxSmfCompatibleModule.audibleFeedback
    api.cboxSmfCompatibleModule.audibleFeedback = False
    for d in events:
        data = d[2] #just the midi stuff.
        if data and len(data) == 3:
            event, midipitch, velocity = data #of course it doesn't need to be pitch and velocity. But this is easier to remember than "data2, data3"
            channel = event & 0x0F
            mode = event & 0xF0

            if mode == 0x90 and api.cboxSmfCompatibleModule.jack_midi_in_is_open:  #the midi in port can be muted: #note on. Note off gets ignored.
                #if api.cboxSmfCompatibleModule.states.cc[0x07]: #volume not 0?
                if api.cboxSmfCompatibleModule.states.digitalCC(0x40): #pedal down?
                    if api.cboxSmfCompatibleModule.states.duringChordEntry: #there was a note before the current one with a pedal down. We are in a chord adding process
                        api.addNoteToChord(api.pitch.midiToPitch(midipitch, api._getKeysig()))
                    else:
                        api.cboxSmfCompatibleModule.states.duringChordEntry = True
                        api.insertPrevailingDuration(api.pitch.midiToPitch(midipitch, api._getKeysig()))
                        api.left() #executeCC for pedal up does the right() function in the end
                else:
                    api.insertPrevailingDuration(api.pitch.midiToPitch(midipitch, api._getKeysig()))
                #else:
                #  api._insertRest(api._getCursorDuration())
            elif mode == 0xB0:  #CC
                #api.cboxSmfCompatibleModule.states.cc[midipitch] = velocity #save the controller data
                if midipitch in api.cboxSmfCompatibleModule.executeCC:
                    #The executeCC function receives the new second midi byte ("Velocity") and the velocity value exactly before this midi byte, so you can compare with the old one.
                    returnfunction = api.cboxSmfCompatibleModule.executeCC[midipitch](velocity, api.cboxSmfCompatibleModule.states.cc[midipitch]) #execute the bound function, if existent.
                    if returnfunction:
                        eval(returnfunction)()
                api.cboxSmfCompatibleModule.states.cc[midipitch] = velocity #save the controller data

    api.cboxSmfCompatibleModule.audibleFeedback = resetSound

    #api.cboxSmfCompatibleModule.transport.status().playing is for the internal state. It does not reflect the jack transport status. Thats why we check our song position and length instead.
    ppqn = api.cboxSmfCompatibleModule.transport.status().pos_ppqn
    if ppqn is None: #Before the first playback this is None (no file) or 0 (no playback). After the first playback it is never None again, even if no file.
        return True
    if (not api.cboxSmfCompatibleModule.keepRolling) and ppqn >= api.cboxSmfCompatibleModule._SMF.lastMaxTicks:
        api.cboxSmfCompatibleModule.transport.stop()
        api.l_send(api.l_score.playbackStop)
    elif api.cboxSmfCompatibleModule.transport.status().playing:
        api.l_send(calfboxGetPlaybackTicks)
    elif not api.cboxSmfCompatibleModule.keepRolling: #and not playing
        api.l_send(api.l_score.playbackStop) #this hides the playback bar. Calfbox itself does not move the playback cursor further than our max ticks, even if transports keeps rolling.
    #else: #playback stopped but keepTransportRolling is True

    if not api.cboxSmfCompatibleModule.transport.status().playing and api._getWorkspace(): #not during playback and only if a file is open.
        api.cboxSmfCompatibleModule.updateMidiInState(api._getActiveCursor()) #TODO: This is most likely a performance problem. But it was at least not visible in htop for now. Not all functions were in the updater then.

def play(startTick = None, workspace = None, jackMode = False, standaloneMode = False, parts = None, waitForStart = False, keepRolling = True):
    """Start the playback through calfbox/midi
    Playback starts at the cursor position in its own thread.
    Does not move the cursor.

    Returns (cbox module, smf/calf-class,
    and maxTicks) which is the end of the song.

    If used in a non-gui script use waitForEnd = True.
    This will set play into a loop and wait until playback has ended
    before continuing the script. Else your script will end before
    playback has ended and shut down the playback.
    """


    if not api.cboxSmfCompatibleModule:
        #A GUI would not allow that but a wrong userscript might still try this.
        raise RuntimeError("You have to initialize calfbox before using playback.")

    if not workspace:
        workspace = api._getWorkspace()
        if not workspace:
            raise ValueError("Parameter workspace must be core.Workspace or core.Collection but it was:", workspace)

    if api.cboxSmfCompatibleModule.transport.status().playing: #Pause
        api.l_send(api.l_score.playbackStop)
        api.cboxSmfCompatibleModule.running = False
        api.cboxSmfCompatibleModule.transport.stop()
        return True

    smfStructure = workspace.score.exportCalfbox(smf = api.cboxSmfCompatibleModule, jackMode = jackMode, parts = parts) # "smfStructure" is an relict from libsmf. We use it here as well to make comparison with exportMidi easier.
    if not smfStructure: #empty Score
        return False
    maxTicks = smfStructure.update(jackMode= jackMode) #convert to binary format and send to RT thread. Returns the overall length of the song.

    # Start playback. From this moment on playback is done in a seperate track.
    #The only important thing is not to close the whole program until the playback has finished.
    #The most simple way to do this is just by letting the main program, Laborejo, sleep.
    #Instead of libsmfs "save()" method for the returned smfStructure/calfData we have smf.play()
    if startTick is None:
        startTick = smfStructure.playbackStartPosition

    api.cboxSmfCompatibleModule.negativePlaybackTickOffset = 0
    api.cboxSmfCompatibleModule.running = True
    api.cboxSmfCompatibleModule.keepRolling = keepRolling

    if not waitForStart:
        api.l_send(api.l_score.playbackStart)
        api.cboxSmfCompatibleModule.play(startTick = startTick)

    if standaloneMode: #Don't call from a GUI!
    #No, don't run from a GUI!
    #This part blocks, has sleep() in it.
        endTick = int(api.cboxSmfCompatibleModule._SMF.lastMaxTicks)
        endMinutes, endSeconds = divmod(api.cboxSmfCompatibleModule._SMF.getSongLengthInSeconds(), 60)
        endTime = "".join((str(int(endMinutes)), ":", str(int(endSeconds)).zfill(2)))
        endMinuteLen = len(str(int(endMinutes)))
        def printStatus():
            status = api.cboxSmfCompatibleModule.transport.status()
            now = status.pos_ppqn
            seconds = status.pos/status.sample_rate
            minutes, seconds = divmod(seconds, 60)
            steps = int(round(now/endTick,2)*30)
            bar = "".join(("\r",'[', steps*"#", (30-steps)*" ", '] ', str(int(minutes)).zfill(endMinuteLen), ":", str(int(seconds)).zfill(2), "/", endTime))
            print (bar, end="")
        while smfStructure.transport.status().pos_ppqn < endTick and api.cboxSmfCompatibleModule.running:
            # Get transport information - current position (samples and pulses), current tempo etc.
            #master = smfStructure.transport.status().pos_ppqn #'pos', 'pos_ppqn', 'tempo', 'timesig', 'sample_rate'
            printStatus()
            smfStructure.cbox.call_on_idle() # Query JACK ports, new USB devices etc.
            time.sleep(0.2)
        printStatus() #a last time because the while loop is inaccurate and ends more or less "randomly" at a certain tick.
        if not keepRolling:
            api.cboxSmfCompatibleModule.transport.stop()
    return (smfStructure, maxTicks) #This enables any frontend or script to query the status. See the loop below:

def playOnce(startTick = 0, jackMode=False, waitForStart = False, keepRolling = True):
    """A tool for the commandline or script files.
    Just play it once, block while its playing."""
    play(startTick = startTick, jackMode=jackMode, standaloneMode = True, waitForStart = waitForStart, keepRolling = keepRolling)

def playCurrentTrack(jackMode = False, waitForStart = False, keepRolling = True):
    play(jackMode=jackMode, parts = ("_uniqueContainerName", api._getTrack().uniqueContainerName ), waitForStart = waitForStart, keepRolling = keepRolling) #core.parts is a tuple (trackProperty, value). Only those which match will be exported.

def playCurrentGroup(jackMode = False, waitForStart = False, keepRolling = True):
    play(jackMode=jackMode, parts = ("group", api._getTrack().group ), waitForStart = waitForStart, keepRolling = keepRolling) #core.parts is a tuple (trackProperty, value). Only those which match will be exported.

def stop():
    if api.cboxSmfCompatibleModule: #several functions like new, load, close call stop to prevent crazy playback behaviour.
        api.cboxSmfCompatibleModule.running = False
        api.cboxSmfCompatibleModule.transport.stop()
        api.l_send(api.l_score.playbackPanic)
        api.cboxSmfCompatibleModule.panic()

def setSoundfont(filepath):
    """Load a new soundfont. Only one at a time."""
    if os.path.exists(filepath):
        api.cboxSmfCompatibleModule.newInternalInstrument(filepath)
    else:
        warnings.warn("Soundfont " + filepath + " does not exist")

def scaleTempo(factor):
    """Scale the tempo. Works during playback.
    Everytime you use factor it will be scaled from 1 again,
    not from the current tempo"""
    api.cboxSmfCompatibleModule._SMF.scaleTempoLive(factor)

def getMidiPorts():
    """Just forwards to the calfbox module. Returns a list of strings"""
    if api.cboxSmfCompatibleModule:
        return api.cboxSmfCompatibleModule.getMidiPorts()
    else:
        return []

def getAudioPorts():
    """Just forwards to the calfbox module. Returns a list of strings"""
    if api.cboxSmfCompatibleModule:
        return api.cboxSmfCompatibleModule.getAudioPorts()
    else:
        return []
