# -*- coding: utf-8 -*-
"""
This file is part of Laborejo - http://www.laborejo.org
Author: Nils Gey info@laborejo.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import laborejocore as api
from laborejocore import items, playback
from copy import deepcopy

def message(message, title = _("Laborejo Message"), function = None):
    """Outputs a simple warning message and waits for the user
    to accept or cancel.
    Can be used in the gui or in textmode.
    Can be used as if statement: if warning("foo").
    If function is provided the function is called to calculate the
    return message. The function needs to accept one bool parameter
    which is the Users Ok/Cancel."""
    if api.STANDALONE:
        raise RuntimeError("Encountered Textmode warning. Not supported. Stopping to prevent damage. Message:", message)
        exit() #TODO: Create textmode question.
    else:
        ret = api.l_form.generateForm({}, title, message)
        if ret is not None:
            if function:
                return function(ret)
            else:
                return True
        else:
            if function:
                return function(ret)
            else:
                return False

def _dict2dynformDict(dictionary, extraTags = None):
    """return a new dict which is compatible to dynformcreator:
    dict[k] = v  -->  dict[k] = (k, v).
    The k in the tuple (k, v) is the printed tag.

    Instead of k you can give extra Tags which will be used instead.
    extraTags key =  original dict key
    extraTags value = printed tag (no other usage)
    """
    new = {}
    if extraTags:
        for k,v in dictionary.items():
            new[k] = (extraTags[k], v)
    else:
        for k,v in dictionary.items():
            new[k] = (k, v)
    return new

def _editDictWithGui(data, title, comment):
    """Generate a gui to edit a dictionary in place.
    Leave field blank to delete the dict entry"""
    result = api.l_form.generateForm(_dict2dynformDict(data), title, comment)
    if result:
        for k,v in x.items():
            if v:
                data[k] = v
            else:
                del data[k]

def _editMultipleDictWitTabbedhGui(data, title):
    """Change multiple dicts in place with a generated, tabbed gui.
    Wants a list of lists:[dict, tab-title, tab-comment].
    DON'T use tuples, only lists"""
    newdata = deepcopy(data) #make a copy
    for x in newdata:
        x[0] = _dict2dynformDict(x[0]) #replace each dict in the copied data with a transformed gui dict. But leave original data intact so we can later change it.

    result = api.l_form.generateForm(newdata, title) #returns a list of dicts

    if result:
        for number, dictionary in enumerate(result):
            for k,v in dictionary.items():
                origDict = data[number][0]
                if v:
                    origDict[k] = v #0 is the dict position, 1 and 2 are gui strings.
                else:
                    del origDict[k]
        return True
    else:
        return False

def _deleteFromDictWithGui(dictionary, title, comment):
    """Create a shadow dict which has the same tags but "False" as value
    then delete all entries from the original dict which were switched
    to True by the user"""
    new = {}
    for k,v in dictionary.items(): #key are strings
        new[k] = (k, False)
    new["0000All"] = ("Delete All", False)

    result = api.l_form.generateForm(new, title, comment)
    if result:
        if result["0000All"]:
            dictionary.clear()
            return True
        else:
            del result["0000All"]
            for k,v in result.items():
                if v:
                    del dictionary[k]
            return True
    else: #Canceled GUI.
        return False

def generateDropDownGuiFromFunctions(functionlist, default, label, title="", comment=""):
    """Creates a gui with a dropdown list to choose a function by
    its pretty string name.
    functionlist is a list of tuplets:
    [(function, "string"), ...]"""
    functionlist.insert(0, default)
    result = api.l_form.generateForm({"singleEntry":(label, functionlist)}, title, comment)
    if result:
        result["singleEntry"]()

def rawEditGui():
    """edit the items python __dict__ directly with a generated GUI.
    Only for debugging and testing. No checks, no safety, no warranty."""

    plainDict = {}
    data = [[plainDict, "General", ""]]
    for key, value in api._getCursorItem().__dict__.items(): #key is the instance variable name and value its value, which might be another dict
        if type(value) is dict:
            data.append([value, key, ""])
        else:
            plainDict[key] = value
    #print()
    #import pprint
    #pprint.pprint (data)

    #_editMultipleDictWitTabbedhGui(data, "Raw Item Edit")
    #_editDictWithGui(plainDict, "bla", "so")

    #TODO: this is broken because fedit is not powerful enough. It can't handle dicts, empty lists etc. etc. don't work around here by filtering data. Work in fedit.


def metadataGui():
    """Provide a gui dialog to ask all lilypond metadata"""
    header = api._getHeader().data

    data = {"00title":(_("Title"), header["title"]),
            "02subtitle":(_("Subtitle"), header["subtitle"]),
            "04dedication":(_("Dedication"), header["dedication"]),
            "06composer":(_("Composer"), header["composer"]),
            "08subsubtitle":(_("Subsubtitle"), header["subsubtitle"]),
            "10poet":(_("Poet"), header["poet"]),
            "12instrument":(_("Instrument"), header["instrument"]),
            "14meter":(_("Meter"), header["meter"]),
            "16arranger":(_("Arranger"), header["arranger"]),
            "18piece":(_("Piece"), header["piece"]),
            "20opus":(_("Opus"), header["opus"]),
            "22copyright":(_("Copyright"), header["copyright"]),
            "24tagline":(_("Tagline"), header["tagline"]),
            "30subtextlabel":(None, _("Subtext")),
            "31subtext":("", api._getScore().subtext),
            "32subtextlabel":(None, _("&lt;br&gt; for a line break. Empty line for new paragraph.\nMore with Lilypond Markup Syntax")),
            }
    result = api.l_form.generateForm(data, _("Metadata"), _("These will be visible in the print out as well.\nAn entry starting with \\markup makes the field treated as literal lilypond code.\nYou have to \"quote\" strings yourself then."))
    if result: #Cancel?
        api._getScore().subtext = result["31subtext"]
        #Now remove the subtexts and loop over the rest
        del result["31subtext"]

        for key, value in result.items():
            header[key[2:]] = value
        return True
    else:
        return False

def directivesGui(backendItem):
    """Provide a gui dialog to review and change all directives"""
    data = [[backendItem.directivePre, "Prefix", _("Before the note:\nprefix<c'>2\n\nLeave field blank to delete the directive.")],
            [backendItem.directiveMid, "Middle", _("Between notes and duration:\n<c'>middle2\n\nLeave field blank to delete the directive.")],
            [backendItem.directivePst, "Postfix", _("After the duration:\n<c'>2postfix\n\nLeave field blank to delete the directive.")],
            [backendItem.instructionPre, "Pre Midi", _("Executed before playing back this item.\nInsert Midibytes as a list.\nLeave field blank to delete the directive.") ],
            [backendItem.instructionPst, "Post Midi", _("Executed after playing back this item.\nInsert Midibytes as a list.\nLeave field blank to delete the directive.") ],
            ]
    if _editMultipleDictWitTabbedhGui(data, "Lilypond Directives"):
        api.l_send(api.l_score.trackChanged)
        api.l_send(api.l_score.trackDirectivesChanged)

     #"Additional Lilypond instructions before, in between and after this item"

def createDirectivesGui():
    """The directive keys get an "usr_" prefix so they don't collide
    with internal keys"""
    data = {"01a":(None, "PREFIX<c>4"),
            "01pre":("Prefix", ""),
            "02a":(None, "<c>MIDDLE4"),
            "02mid":("Middle", ""),
            "03a":(None, "<c>4POSTFIX"),
            "03pst":("Postfix", ""),
            "04midipre":("Midi Pre Instruction", ""),
            "05midipst":("Midi Post Instruction", ""),
            }
    ret = api.l_form.generateForm(data, _("Create Object Directives"), _("Leave fields blank to not create any directive of this kind." ))
    if ret: #cancel?
        item = api._getCursorItem()
        if ret["01pre"]: #not empty
            item.directivePre["usr_" + str(random.randrange(1,1000))] = ret["01pre"]
        if ret["02mid"]: #not empty
            item.directiveMid["usr_" + str(random.randrange(1,1000))] = ret["02mid"]
        if ret["03pst"]: #not empty
            item.directivePst["usr_" + str(random.randrange(1,1000))] = ret["03pst"]
        if ret["04midipre"]: #not empty
            item.instructionPre["usr_" + str(random.randrange(1,1000))] = ret["04midipre"]
        if ret["05midipst"]: #not empty
            item.instructionPst["usr_" + str(random.randrange(1,1000))] = ret["05midipst"]
        api.l_send(lambda: api.l_item.update(item))

def editGui():
    """A general purpose edit that will detect what item is under
    the cursor and call the proper function.
    No advanced functionality like undo in here. This is just a
    convenience if/elif function"""
    types = {
            items.Chord : editDirectives,
            items.PerformanceSignature : editPerformanceSignature,
            items.Markup : editMarkup,
            items.Chord : editDirectives,
            items.RepeatClose : editRepeat,
            items.RepeatCloseOpen : editRepeat,
            items.GotoCapo : editDaCapoDalSegno,
            items.GotoSegno : editDaCapoDalSegno,
            items.Start : editContainer,
            items.End : editContainer,
            items.AlternateEnd : editAlternateEnd,
            items.ChannelChangeRelative : editChannelChangeRelative,
            items.ChannelChangeAbsolute : editChannelChangeAbsolute,
            items.ProgramChangeRelative : editProgramChangeRelative,
            items.ProgramChangeAbsolute : editProgramChangeAbsolute,
            items.TempoModification : editTempoModification,
            items.DynamicSignature : editDynamicSignature,
            items.InstrumentChange : editInstrumentChange,
            items.TempoSignature : editTempoSignature ,
            api.core.Container : editContainer,
            }

    typ = type(api._getCursorItem())
    if typ in types:
        types[typ]()
    else:
        return False


def editContainer():
    typ = type(api._getCursorItem())
    if typ is api.core.Container:
        cont = api._getCursorItem()
    elif typ is items.Start:
        cont = api._getCursorItem().parentContainer
    else:
        cont = api._getContainer()
    if cont:
        #ret = api.l_form.generateForm({"repeatPercent":(_("Percent-Repeat incl. first"), cont.repeatPercent)}, _("Edit Container: " + cont.uniqueContainerName), "")
        ret = api.l_form.generateForm({"repeatPercent":(_("Percent-Repeat incl. first"), cont.repeatPercent), "cursorWalk":(_("Cursor Walk"), cont.cursorWalk), "uniqueContainerName":(_("Unique Name"), cont.uniqueContainerName), },  _("Edit Container: ") + cont.uniqueContainerName)
        if ret:
            cont.repeatPercent = ret["repeatPercent"]
            cont.uniqueContainerName = ret["uniqueContainerName"]
            #cont.uniqueContainerName = ret["uniqueContainerName"]
            walkChanged = False if cont.cursorWalk == ret["cursorWalk"] else True
            cont.cursorWalk = ret["cursorWalk"]
            if walkChanged:
                #This is a big change. Trigger a full gui redraw #TODO: currently. This is very qt gui specific.
                api.l_send(api.l_score.fullResetAndRedraw)
                return True #and exit this function

            api.l_send(lambda: api.l_item.updateStatic(cont))
            api.l_send(lambda: api.l_item.updateDynamic(cont))
            api.l_send(lambda: api.l_item.updateStatic(cont.container[0]))
            api.l_send(lambda: api.l_item.updateStatic(cont.container[-1]))

            api.syncCursors()
            api.l_send(lambda: api.l_cursor.setPosition(api._getWorkspace(), api._getTrackIndex(), api._getFlatCursorIndex()))
            if cont.cursorWalk:
                api.l_send(lambda: api.l_score.updateTrackFromItem(cont[-1])) #update from the End marker, which maybe has a new duration
            else:
                api.l_send(lambda: api.l_score.updateTrackFromItem(cont)) #update from the Container istself, which maybe has a new duration

def guiChooseContainer():
    """Gui Only. Offer a choice from existing containers."""
    dictSortedAfterName = {k.uniqueContainerName:k for k,v in api._getScore().nonTrackContainers.items()}
    autolist = list(dictSortedAfterName.keys())
    ret = api.l_form.generateForm({"text":("","")}, "Insert Container", "Type in Container name", autocomplete=autolist)
    if ret and ret["text"] in autolist:
        cont = dictSortedAfterName[ret["text"]]
        api.insertContainer(cont)


def editTrigger():
    item = api._getCursorItem()
    start = item.triggerString
    if start is None:
        start = ""
    ret = api.l_form.generateForm({"triggerString":("", start)}, _("Edit Playback Trigger"), _("You must specify a variable 'ret' as True or False.\nThis will be used to determin if this item will be played back or not."))
    if ret:
        api.saveState(item, api.l_item.smfTrigger, durationChanged = False)
        item.triggerString = ret["triggerString"]
    api.l_send(lambda: api.l_item.smfTrigger(item))

def editDirectives():
    directivesGui(api._getCursorItem())

def editStandalone():
    """The edit variants are gui only. Any script can directly access .lilypond"""
    ret = api.l_form.generateForm({"standalone":(" ", str(api._getCursorItem().lilypond))}, _("Standalone Lilypond Directive"), _("Lilypond text to insert on this position.\n\nThis will override any generated (notes, clefs etc.) text.\nAn empty string, no spaces, will enable auto-generation again."))
    if ret:
        api._getCursorItem().lilypond = ret["standalone"]

def editStandaloneWaitForNextChord():
    """The edit variants are gui only. Any script can directly access .lilystring"""
    ret = api.l_form.generateForm({"standalone":(" ", str(api._getCursorItem().lilystring))}, _("Wait-For-Next-Chord Lilypond Directive"), _("The text will be used as post directive for the next chord/note."))
    if ret:
        api._getCursorItem().lilystring = ret["standalone"]

def editPerformanceSignature(default=False):
    """Create a tabbed gui edit which modifies all perf-sig values.
    if default = False it will edit the default one instead of
    the cursor one"""
    if default:
        item = api._getScore().defaultPerformanceSignature
    else:
        item = api._getCursorItem()

    if type(item) is items.PerformanceSignature:
        data1 = {
                "00name":(_("Optional: Name"), item.name),
                "00r" : (None, ""),
                "00staccato":(_("Staccato"), item.staccato),
                "00z-tenuto":(_("Tenuto"), item.tenuto),
                "01legatoScaling":(_("Legato (Slurs)\nSmaller value means shorter duration. Must be bigger than 0. Full Length: 1. "), item.legatoScaling),
                "01slurEndScaling":(_("Slur End\nSmaller value means shorter duration. Must be bigger than 0. Default: 0.5. "), item.slurEndScaling),
                "01nonLegatoScaling":(_("Non Legato/Detache\nSmaller value means shorter duration.\nSet to 0 to activate scalings by note duration (see Tab)."), item.nonLegatoScaling),
                "02nonLegatoHumanizing":(_("Non Legato Humanizing\n1 means no change.\n<1 means a random chance to shorten the duration (e.g. 0.75)\n>1 means a random chance to extend the duration. (e.g. 1.25)"), item.nonLegatoHumanizing),
                "03z" : (None, _("\nFermata values modify the tempo.\nSmaller value means longer fermata.")),
                "04shortfermata": (_("Short Fermata"), item.shortfermata),
                "05fermata": (_("Fermata"), item.fermata),
                "06longfermata": (_("Long Fermata"), item.longfermata),
                "07verylongfermata": (_("Very long Fermata"), item.verylongfermata),
                }

        velocities = {
                "00main":(_("Main Stress Velocity\nRecommended: 1:"), item.velocityMetricalMain),
                "01secondary":(_("Semi-stressed\nExample:0.95"), item.velocityMetricalSecondary),
                "02else":(_("Non-stressed\nExample:0.85"), item.velocityMetricalNothing),
                "03velocityHumanizing": (_("Velocity Humanizing\nSmaller value means more random velocity substraction. "), item.velocityHumanizing),
                }

        swing = {
                "01swingMod":(_("Swing Modificator\n0 to deactivate."), item.swingMod),
                "02swingLayer":(_("Rhythm Layer for Swing"), [item.swingLayer, (1536*4, _("1 Longa")),(1536*2, _("1 Whole / Semibreve")),(1536, _("1 Whole / Semibreve")), (768, _("2 Half / Minim")), (384, _("4 Quarter / Crotchet")), (192, _("8 Eigth / Quaver")), (96, _("16 Sixteenth / Semiquaver")), (48, _("32 Thirthy-Second / Demisemiquaver")), (24, _("64 Sixty-Fourth / Hemidemisemiquaver")), (12, _("128 Hundred Twenty-Eighth / Semihemidemisemiquaver ")), (6, _("256 Twohundred Fifty-Sixth / Demisemihemidemisemiquaver"))]),
                }

        dynamics = _dict2dynformDict(deepcopy(item.dynamics)) #replace each dict in the copied data with a transformed gui dict. But leave original data intact so we can later change it.
        nonLegatoScalingByDuration = _dict2dynformDict(deepcopy(item.nonLegatoScalingByDuration), extraTags = {12288 : "Maxima", 6144 : "Longa", 3072 : "Breve", 1536 : "Whole", 768 : "Half", 384 : "Quarter", 192 : "Eighth", 96 : "Sixteenth", 48 : "1/32", 24 : "1/64", 12 : "1/128", 6 : "1/256"})

        listoflists = [
            #Keep that list in this order! We use the indeces in the result processing.
            [data1, "Performance", _("These values influence the midi generation.")],
            [dynamics, "Dynamics", _("What midi values (0-127) should be used for dynamics?")],
            [velocities, "Velocities", _("Between a factor 0 and 1, how loud are different metrical positions. Use values near 1 for a realistic effect.")],
            [nonLegatoScalingByDuration, _("Non Legato Scalings"), _("These values are only used if the main Non Legato/Detache value is set to 0.\nChoose how long each base duration will be scalen.\nSmaller value means shorter duration.")],
            [swing, "Swing", _("How much swinging do you want?\nValues must be greater than -3 and smaller than 3 (not included)\n\n0 means binary rhythm (Default)\n1 is strict swing 2/3 + 1/3\n-1 is inverse swing. 1/2 + 2/3\nRecommended: 0.75 to 1.25 (or negative, for inverse swinging) ")],
            ]
        result = api.l_form.generateForm(listoflists, _("Edit Performance Signature")) #returns a list of dicts

        if result: #pressed cancel?
            api.saveState(item, api.l_item.update, durationChanged = False)
            #Most of these values must be forced to be a float value. Otherwise the formgen will put ints in there and you never get floats again via the gui.
            item.dynamics = result[1] #the second returned dict is just in the right format. We can use it to overwrite the original data.
            for k,v in result[3].items(): #convert ints to floats again
                result[3][k] = float(v)
            item.nonLegatoScalingByDuration = result[3] #the second returned dict is just in the right format. We can use it to overwrite the original data.
            item.name = result[0]["00name"]
            item.staccato = float(result[0]["00staccato"])
            item.tenuto = float(result[0]["00z-tenuto"])
            item.nonLegatoScaling = float(result[0]["01nonLegatoScaling"])
            item.legatoScaling = float(result[0]["01legatoScaling"])
            item.slurEndScaling = float(result[0]["01slurEndScaling"])
            item.nonLegatoHumanizing = float(result[0]["02nonLegatoHumanizing"])
            item.shortfermata = float(result[0]["04shortfermata"])
            item.fermata = float(result[0]["05fermata"])
            item.longfermata = float(result[0]["06longfermata"])
            item.verylongfermata = float(result[0]["07verylongfermata"])

            item.velocityMetricalMain = float(result[2]["00main"])
            item.velocityMetricalSecondary = float(result[2]["01secondary"])
            item.velocityMetricalNothing = float(result[2]["02else"])
            item.velocityHumanizing = float(result[2]["03velocityHumanizing"])

            item.swingMod = float(result[4]["01swingMod"])
            item.swingLayer = int(result[4]["02swingLayer"])
            api.l_send(lambda: api.l_item.update(item))


def editInstrumentChange():
    item = api._getCursorItem()
    if type(item) is items.InstrumentChange:
        data0 = {
                "00a" : (None, _("Short Instrument is used for print-out.")),
                "00longInstrumentName": (_("Long Instrument Name"), item.longInstrumentName),
                "01shortInstrumentName": (_("Short Instrument Name"), item.shortInstrumentName),
                }
        data1 = {
                "02a" : (None, _("\nAll Midi values are 1-128")), #+1 for user display. 0-127 is original
                "02smfPatch":(_("Midi Instrument"), [item.smfPatch] + playback.instrumentList),
                "03smfVolume":(_("Midi Volume"), item.smfVolume + 1),
                "03z" : (None, _("\nAdvanced:")),
                "04smfBank":(_("Midi Bank"), item.smfBank + 1),
                }
        data2 = {
                "02a" : (None, _("\nAll values are 1-128.\n0 to deactivate")), #+1 for user display. 0-127 is original
                "02jackPatch":(_("Patch"), item.jackPatch +1),
                "03jackVolume":(_("Volume"), item.jackVolume + 1),
                "04jackBank":(_("Bank"), item.jackBank + 1),
                }
        listoflists = [
            [data0, "Meta-Data", _("Internal names as well as PDF and Lilypond output")],
            [data1, "Internal Sound", _("These values influence the internal sound output.")],
            [data2, "JACK Sound", _("These values influence the JACK sound output.")],
            ]
        result = api.l_form.generateForm(listoflists, _("Edit Instrument Change")) #returns a list of dicts

        if result: #pressed cancel? no:
            api.saveState(item, api.l_item.update, durationChanged = False)
            item.longInstrumentName = result[0]["00longInstrumentName"]
            item.shortInstrumentName = result[0]["01shortInstrumentName"]
            item.smfPatch = playback.instrumentList.index(result[1]["02smfPatch"])
            item.smfVolume = result[1]["03smfVolume"] -1 #user values are one too much for me! :(
            item.smfBank = result[1]["04smfBank"] -1
            item.jackPatch = result[2]["02jackPatch"] -1
            item.jackVolume = result[2]["03jackVolume"] -1
            item.jackBank = result[2]["04jackBank"] -1
            api.l_send(lambda: api.l_item.update(item))


def editMarkup():
    """The edit variants are gui only. Any script can directly access .lilystring and .position = ^ or _"""
    cursorItem = api._getCursorItem()
    ret = api.l_form.generateForm({"lilystring":(_("Text"), str(cursorItem.lilystring)), "position":(_("Position"), [cursorItem.position, ("^", _("above")), ("_", "under")])}, _("Free Text"), _("Enter any text and choose above or under the track.\n\nYou can use Lilypond markup syntax"))
    if ret: #don't accept empty strings. This item can easily be deleted.
        api.saveState(cursorItem, api.l_item.update, durationChanged = False)
        i =api._getCursorItem()
        i.lilystring = ret["lilystring"]
        i.position = ret["position"]
        api.l_send(lambda: api.l_item.update(i))

def editTempoSignature():
    cursorItem = api._getCursorItem()

    #there are only two possibilities for standard tempo sigs. Either the tempo is dotted or not.
    if cursorItem.referenceTicks in (1536*4, 1536*2, 1536, 768, 384, 192, 96, 48, 24, 12, 6):
        realReferenceTicks = int(cursorItem.referenceTicks)
        dotted = False
    else:
        realReferenceTicks = int(cursorItem.referenceTicks / 1.5)
        dotted = True

    data = {"01bpm": (_("Beats per Minute"), cursorItem.beatsPerMinute),
        "02reference": (_("Reference Note"), [realReferenceTicks, (1536*4, _("1 Longa")),(1536*2, _("1 Whole / Semibreve")),(1536, _("1 Whole / Semibreve")), (768, _("2 Half / Minim")), (384, _("4 Quarter / Crotchet")), (192, _("8 Eigth / Quaver")), (96, _("16 Sixteenth / Semiquaver")), (48, _("32 Thirthy-Second / Demisemiquaver")), (24, _("64 Sixty-Fourth / Hemidemisemiquaver")), (12, _("128 Hundred Twenty-Eighth / Semihemidemisemiquaver ")), (6, _("256 Twohundred Fifty-Sixth / Demisemihemidemisemiquaver"))]),
        "02z" : (None, _("Optional:")),
        "03dotted": (_("Dotted"), dotted),
        "04tempostring": (_("Tempo string"), cursorItem.tempostring),
        }
    ret = api.l_form.generateForm(data, _("Tempo Signature"), _("Choose how many (bpm) of which note (reference) per minute. Optional string for printing."))
    if ret:
        cursorItem.beatsPerMinute = ret["01bpm"]
        cursorItem.referenceTicks = ret["02reference"]
        if ret["03dotted"]:
            cursorItem.referenceTicks = int(cursorItem.referenceTicks * 1.5)
        cursorItem.tempostring = ret["04tempostring"]
        api.l_send(lambda: api.l_item.updateStatic(cursorItem))
    else:
        return False

def editRepeat():
    """Access to the number of repeats"""
    ret = api.l_form.generateForm({"repeat":(" ", api._getCursorItem().repeat)}, _("Edit Repeat"), _("Number of Repeats.\nThe first one is not included. Default is 1."))
    if ret:
        i = api._getCursorItem()
        i.repeat = ret["repeat"]
        api.l_send(lambda: api.l_item.updateStatic(i))

def editChannelChangeRelative():
    ret = api.l_form.generateForm({"channel":(" ", api._getCursorItem().number)}, _("Edit Relative Channel Change"), _("Give a number, negative or positve,\n which gets added to the midi channel for notes from here on."))
    if ret or ret == 0:
        i = api._getCursorItem()
        i.number = ret["channel"]
        api.l_send(lambda: api.l_item.updateStatic(i))
    else:
        return False

def editChannelChangeAbsolute():
    ret = api.l_form.generateForm({"channel":(" ", api._getCursorItem().number + 1 )}, _("Edit Absolute Channel Change"), _("Give a number 1-16,\n which is the new midi channel for notes from here on."))
    if ret or ret == 0:
        i = api._getCursorItem()
        i.number = ret["channel"] - 1
        api.l_send(lambda: api.l_item.updateStatic(i))
    else:
        return False

def editProgramChangeRelative():
    ret = api.l_form.generateForm({"program":(" ", api._getCursorItem().number)}, _("Edit Relative Program Change"), _("Give a number, negative or positve,\n which gets added to the midi program for notes from here on."))
    if ret or ret == 0:
        i = api._getCursorItem()
        i.number = ret["program"]
        api.l_send(lambda: api.l_item.updateStatic(i))
    else:
        return False

def editProgramChangeAbsolute():
    ret = api.l_form.generateForm({"program":(" ", api._getCursorItem().number + 1 )}, _("Edit Absolute Program Change"), _("Give a number 1-128,\n which is the new midi program for notes from here on."))
    if ret or ret == 0:
        i = api._getCursorItem()
        i.number = ret["program"] - 1
        api.l_send(lambda: api.l_item.updateStatic(i))
    else:
        return False

def editTempoModification():
    ret = api.l_form.generateForm({"mod":(" ", api._getCursorItem().bpmModificator )}, _("Edit Tempo Modification"), _("Give a positive or negative number to change the bpm relative to the current tempo signature.\n0 is 'A Tempo' and resets any changes."))
    if ret or ret == 0:
        i = api._getCursorItem()
        i.bpmModificator = ret["mod"]
        api.l_send(lambda: api.l_item.updateStatic(i))
    else:
        return False


def editAlternateEnd():
    """Access to the alternate endings."""
    ret = api.l_form.generateForm({"endings":(" ", " ".join([str(x) for x in api._getCursorItem().endings]))}, _("Edit Alternate End"), _("Edit alternate endings. Seperate with space."))
    if ret:
        endings = [int(x) for x in ret["endings"].split()]
        i = api._getCursorItem()
        i.endings = endings
        api.l_send(lambda: api.l_item.updateStatic(i))

def editDaCapoDalSegno():
    ret = api.l_form.generateForm({"disableRepeat":(_("Disable Repeat"), api._getCursorItem().disableRepeat),
                                "until":(_("Play until"), [api._getCursorItem().until, (None, _("End")), ("Fine", "Fine"), ("Coda", "Coda")])},
                                _("Goto marking"), _("Choose if you want to disable repeat during playback after this jump.\nAlso if you want to stop playback before the end."))
    if ret: #don't accept empty strings. This item can easily be deleted.
        i =api._getCursorItem()
        i.disableRepeat = ret["disableRepeat"]
        i.until = ret["until"]
        api.l_send(lambda: api.l_item.update(i))

def editDynamicSignature():
    ret = api.l_form.generateForm({"dyn":(" ", str(api._getCursorItem()._expression) )}, _("Edit Dynamic Signature"), _("Give either a supported string like 'p' or 'f' (see Performance Signature)\nor directly a number (integer) between 0 and 127 (silence and max dynamic)\n\nThe printout will not change."))
    if ret: #even the 0 is a string and soooo True.
        i = api._getCursorItem()
        try: #see if it is a number
            i._expression = int(ret["dyn"])
        except: #it must be a string. If it is false the PerfSig and playback export will tell.
            i._expression = ret["dyn"]
        api.l_send(lambda: api.l_item.updateStatic(i)) #ironically the dynamic signature is update by the static function.
    else:
        return False


def chordSymbolMassInsert():
    """Gui only"""
    while api.chordSymbol(): #the normal gui dialog returns False or None on cancel
        if not api.right(): #right is executed in any case except user cancelled chordSymbol
            break #end of track
        #while not (type(api._getCursorItem()) is items.Rest or type(api._getCursorItem()) is items.Chord):
        while (not isinstance(api._getCursorItem(), items.Chord)) and (not isinstance(api._getCursorItem(), items.MultiMeasureRest)):
            if not api.right():
                return False

def figuredBassMassInsert():
    """Gui only"""
    while api.figuredBass(): #the normal gui dialog returns False or None on cancel
        if not api.right(): #right is executed in any case except user cancelled the gui
            break #end of track
        while not type(api._getCursorItem()) is items.Chord:
            if not api.right():
                return False
